"""
parser.py
~~~~~~~~~

*vxapi* is an unofficial Python client for Virgin America Elevate accounts.

This module provides the capabilities to "understand" Virgin America's unfortunately designed web service.

:copyright: (c) 2014 by Anthony Wu.
:license: MIT License, see LICENSE for more details.
"""

import bs4
import collections
import re
import vxapi.utils as utils
import vxapi.errors as errors

class ParserException(errors.VXException):
    pass

class UnexpectedContentException(ParserException):
    pass

# === "My Account" --> "Elevate Activity"
ElevateAccount = collections.namedtuple('ElevateAccount', [
    'member_name',
    'member_number',
    'points',
    'status'
])

def parse_member_info(page_content):
    soup = bs4.BeautifulSoup(page_content)
    table = soup.find('table', attrs={'id': 'memberInfo'})
    """
    Use the class="red" table cells inside of the memberInfo as a signal for value cell

        <td class="red">John Smith</td>,
        <td class="red">1234567890</td>,
        <td class="red">9,021</td>,
        <td class="red">Red</td>
    """
    try:
        tds = table.find_all('td', attrs={'class': 'red'})
        name, n, p, s = tuple([utils.sanitize_text(each.text) for each in tds])
        p = int(p.replace(',', ''))
        return ElevateAccount(
            member_name=name,
            member_number=n,
            points=p,
            status=s
            )
    except ValueError:
        raise UnexpectedContentException(table)

ActivityItem = collections.namedtuple('ActivityItem', [
    'activity',
    'date',
    'description',
    'points',
    'status'
])

def parse_activity_row(row_content):
    """
    Parse a <tr> that contains a tuple of Elevate *ActivityItem* within its <td>s
    """
    if row_content.find('td', text='No activity found.'):
        return None
    tds = row_content.find_all('td')
    try:
        a, dt, desc, p, s = tuple([utils.sanitize_text(td.text) for td in tds])
    except ValueError:
        raise UnexpectedContentException(row_content)
    p = utils.remove_whitespace(p)
    desc = utils.remove_whitespace(desc)
    return ActivityItem(
        activity=a,
        date=dt,
        description=desc,
        points=p,
        status=s
        )

def parse_paginator(nav_table):
    """
    Parse a table that contains a paginator navigation element, below the Activities table.

    Find a tag like

    <td style="border:none;">Showing 11-20 of 46</td>

    Returns a 3-tuple of (page_start, page_end, and total)
    """
    paginator_text = None
    for td in nav_table.find_all('td'):
        tt = td.text.strip()
        if tt.startswith('Showing'):
            paginator_text = tt
            break
    non_digits = re.compile('\D+')
    numbers = filter(None, non_digits.sub(',', paginator_text).split(','))
    page_start, page_end, total = tuple(numbers)
    return page_start, page_end, total

def parse_activity_page(page_content):
    """
    Parse the "Elevate Activity" account page for member info and list of activities
    """
    soup = bs4.BeautifulSoup(page_content)
    activity_table = soup.find('table', {'id': 'activityTbl'})
    activity_rows = activity_table.find_all('tr')
    activity_list = []
    has_next_page = None
    for each_row in activity_rows:
        activity_item = parse_activity_row(each_row)
        if activity_item:
            activity_list.append(activity_item)
        else:
            has_next_page = False

    if has_next_page == None:
        nav_table = activity_table.find_next('table')
        page_start, page_end, total = parse_paginator(nav_table)
        has_next_page = (page_end < total)

    return activity_list, has_next_page

# === "My Account" --> "My Contact Info"

PersonalInfo = collections.namedtuple('PersonalInfo', [
    'username',
    'member_number',
    'email',
    'dob',
    'redress_number',
    'known_traveler_number'
])

PersonalContactItem = collections.namedtuple('PersonalContactItem', [
    'name',
    'info'
])


def parse_contact_info(page_content):
    """
    Parse the "My Contact Info" account page for member contact information

    TODO: many more elements are available in this doc
    """
    soup = bs4.BeautifulSoup(page_content)
    div_pi = soup.find('div', {'id': 'divPI'})

    fields = {
        # label_td text -> (namedtuple arg name, parse first line of content only)
        'Member #': ('member_number', False),
        'User Name': ('username', False),
        'Email Address': ('email', False),
        'Date of Birth': ('dob', True),
        'Redress #': ('redress_number', False),
        'Known Traveler #': ('known_traveler_number', False)
    }

    personal_info_dict = {}
    for label_text, (key, first_line_only) in fields.iteritems():
        value = utils.find_value(div_pi, label_text + ':', sibling_distance=1, first_line_only=first_line_only)
        if key == 'member_number':
            value = value.split(' ')[0]
        personal_info_dict[key] = value
    return PersonalInfo(**personal_info_dict)
