"""
api.py
~~~~~~

*vxapi* is an unofficial Python client for Virgin America Elevate accounts.

This module provides the main interface to Virgin America's "web service".

:copyright: (c) 2014 by Anthony Wu.
:license: MIT License, see LICENSE for more details.
"""

import datetime
import logging
import requests
import vxapi.errors as errors
import vxapi.parser as parser

logger = logging.getLogger(__name__)

def generate_dummy():
    """
    Mimic this behavior from https://www.virginamerica.com/scripts/GlobalScriptFile.js

        var thetime=new Date();
        var dummy=""+thetime.getHours()+thetime.getMinutes()+thetime.getSeconds();
    """
    now = datetime.datetime.now()
    return '{0}{1}{2}'.format(now.hour, now.minute, now.second)


class VXAuthRequired(errors.VXException):
    pass

class VXAuthError(errors.VXException):
    pass

class VirginAmericaWebSession(requests.sessions.Session):

    DEFAULT_HEADERS = {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.9; rv:28.0) Gecko/20100101 Firefox/28.0'
    }
    BASE_URL = 'https://www.virginamerica.com'
    SIGN_IN_URL = '/signInTravellerAndPurchase.do'
    CONTACT_INFO_PATH = '/showPreferences.do?method=showPrefrences&forwardPage=contactInfo'
    ACTIVITY_PAGE_PATH_P1 = '/showPreferences.do?method=showPrefrences&forwardPage=elevateActivity'
    ACTIVITY_PAGE_PATH_PN = '/nextAndPreviousElevateActivityAction.do?method=nextAndPreviousElevateActivityInfo&moveType={move_type}&currentPageNo={from_page_num}'

    def __init__(self, username, password):
        requests.sessions.Session.__init__(self)
        self.headers.update(self.DEFAULT_HEADERS)
        self._username = username
        self._password = password
        self._authed = False

    def request(self, method, url, *pargs, **kwargs):
        """
        Simple wrapper to detect redirect to session logout page
        """
        resp = requests.sessions.Session.request(self, method, url, *pargs, **kwargs)
        if resp.history:
            h = resp.history[0]
            if h.status_code == 302 and h.url == url:
                raise VXAuthRequired('Authentication Error - url: {0} was redirected to homepage.'.format(url))
        return resp

    @classmethod
    def get_url(cls, path, append_dummy=False):
        url = '{0}{1}'.format(cls.BASE_URL, path)
        if append_dummy:
            url += '&dummy={0}'.format(generate_dummy())
        return url

    def authenticate(self):
        auth_url = self.get_url(self.SIGN_IN_URL)
        post_data = {
            'loginId': self._username,
            'password': self._password,
            # default and dummy values
            'divName': 'null',
            'fromRightBar': 'yes',
            'method': 'signInFromTravellerAndPurchase',
            'fltMethod': 'null'
        }
        try:
            auth_resp = self.post(auth_url, data=post_data)
        except VXAuthRequired as e:
            raise VXAuthError('Incorrect username and password.')
        self._authed = True
        return self

    def get_elevate_activity(self):
        if not self._authed:
            raise VXAuthRequired('Elevate Activity requires authentication')

        account = None
        activities = []
        cur_page = 1

        while True:
            if cur_page == 1:
                page_path = self.ACTIVITY_PAGE_PATH_P1
                url = self.get_url(page_path, append_dummy=True)
            else:
                # other pages are available from the 'next' action of the previous page
                move_type = 'next'
                from_page_num = cur_page - 1
                page_path = self.ACTIVITY_PAGE_PATH_PN.format(move_type=move_type, from_page_num=from_page_num)
                url = self.get_url(page_path, append_dummy=True)

            resp = self.post(url) # yup, it's a POST call for data retrieval, #WTF

            if not account:
                # only need to parse for account info once for all pages
                account = parser.parse_member_info(resp.content)

            page_activities, has_next_page = parser.parse_activity_page(resp.content)
            activities.extend(page_activities)

            if not has_next_page:
                break
            else:
                cur_page += 1

        return account, activities

    def get_contact_info(self):
        if not self._authed:
            raise VXAuthRequired('Contact Info requires authentication')

        url = self.get_url(self.CONTACT_INFO_PATH, append_dummy=True)
        resp = self.get(url)
        return parser.parse_contact_info(resp.content)
