import logging
from datetime import datetime
# http://micheles.googlecode.com/hg/decorator/documentation.html
import decorator
# Docs as per
# http://docs.python.org/release/2.6.5/library/json.html#module-json
import simplejson

from pylons import request, response, session, tmpl_context as c, url
from pylons.controllers.util import abort, redirect

from wdmmg.lib.base import BaseController, render
from wdmmg import model

log = logging.getLogger(__name__)

# List of actions that can be performed on a domain object.
READ = u'READ'


# :TODO: Consider moving to lib/ ?
def jsonpify(func, *args, **kwargs):
    """A decorator that reformats the output as JSON; or, if the
    *callback* parameter is specified (in the HTTP request), as JSONP.
    
    Very much modelled after pylons.decorators.jsonify .
    """

    from pylons.decorators.util import get_pylons

    pylons = get_pylons(args)
    data = func(*args, **kwargs)
    if isinstance(data, (list, tuple)):
        msg = "JSON responses with Array envelopes are susceptible to " \
              "cross-site data leak attacks, see " \
              "http://pylonshq.com/warnings/JSONArray"
        warnings.warn(msg, Warning, 2)
        log.warning(msg)
    result = simplejson.dumps(data)
    if 'callback' in pylons.request.params:
        pylons.response.headers['Content-Type'] = 'text/javascript'
        log.debug("Returning JSONP wrapped action output")
        # The parameter is a unicode object, which we don't want (as it
        # causes Pylons to complain when we return a unicode object from
        # this function).  All reasonable values of this parameter will
        # "str" with no problem (ASCII clean).  So we do that then.
        cbname = str(pylons.request.params['callback'])
        result = '%s(%s);' % (cbname, result)
    else:
        pylons.response.headers['Content-Type'] = 'application/json'
        log.debug("Returning JSON wrapped action output")
    return result
jsonpify = decorator.decorator(jsonpify)


class RestController(BaseController):

    def index(self):
        dataset_ = model.Session.query(model.Dataset).first()
        enumeration_value = model.Session.query(model.EnumerationValue).first()
        key = enumeration_value.key
        c.urls = [
            url(controller='rest', action='dataset', name_or_id=dataset_.name),
            url(controller='rest', action='dataset', name_or_id=dataset_.id),
            url(controller='rest', action='entry',
                id_=model.Session.query(model.Entry).first().id),
            url(controller='rest', action='key', name_or_id=key.name),
            url(controller='rest', action='key', name_or_id=key.id),
            url(controller='rest', action='enumeration_value',
                name_or_id=enumeration_value.key.name, code=enumeration_value.code),
        ]
        return render('home/rest.html')
    
    @jsonpify
    def dataset(self, name_or_id=None):
        return self._domain_object(self.get_by_name_or_id(model.Dataset, name_or_id))
        
    @jsonpify
    def entry(self, id_=None):
        return self._domain_object(self.get_by_id(model.Entry, id_))
        
    @jsonpify
    def key(self, name_or_id=None):
        return self._domain_object(self.get_by_name_or_id(model.Key, name_or_id))
        
    @jsonpify
    def enumeration_value_id(self, id_=None):
        '''Deprecated'''
        return self._domain_object(self.get_by_id(model.EnumerationValue, id_))
        
    @jsonpify
    def enumeration_value(self, name_or_id=None, code=None):
        '''
        name_or_id - a `Key.name or a `Key.id`.
        code - an `EnumerationValue.code`.
        '''
        key = self.get_by_name_or_id(model.Key, name_or_id)
        ev = (model.Session.query(model.EnumerationValue)
            .filter_by(key=key)
            .filter_by(code=code)
            ).first()
        return self._domain_object(ev)

    def _domain_object(self, domain_object):
        self._check_access(domain_object, READ)
        return domain_object.as_big_dict()

    def _check_access(self, domain_object, action):
        '''
        Checks whether the supplied `apikey` permits `action` to be performed
        on `domain_object`. If allowed, returns `True`. If forbidden, throws
        an appropriate HTTPException.
        '''
        if action == READ:
            return True
        abort(403)
