import csv
from StringIO import StringIO

# National statistics for tax paid per household income decile, 2008/9.
# Taken from Table 14 in http://www.statistics.gov.uk/CCI/article.asp?ID=2440
# Row 1 is average gross income per decile - including benefits, pensions etc
# Row 2 is average direct taxation - income tax, employee NI and council tax
# Row 3 is average indirect taxation - VAT, alcohol, tobacco, petrol etc
# We connect the data points by linear interpolation.
reader = csv.reader(StringIO('''\
0,9219,13583,17204,22040,25190,32995,37592,46268,56889,94341
0,1172,1368,1939,3108,3973,6118,7423,10172,13463,23047
0,2904,2820,3292,3892,3964,4668,5104,6054,6613,8095
'''))
income_table = list(reader)
income_table = [[(float(income)) for income in row] for row in income_table]

class TaxCalculator2010(object):
    def total_tax(self, 
          income, 
          spending=None,
		  is_smoker=False,
		  is_driver=True,
		):
        '''Estimates a person's tax contribution based on the following
        information.
        
        :param income: Total household income. This is used to estimate both
          direct and indirect tax paid.
	    :param spending: Household expenditure. This is used to estimate VAT paid.
		  No longer used, but included for back-compatibility.
		:param is_smoker: - `True` if the person smokes, or `False` if not. Not
		  currently used, but included for back-compatibility.
	    :param is_driver: - `True` if the person drives a car, or `False` if not. Not
		  currently used, but included for back-compatibility.
        
        :returns: a pair `(total_tax, explanation)`. The `explanation` is a list of
            strings describing the steps of the calculation.
        '''
        # Calculate which income range the income falls in.
        lower = 0.0
        income = float(income)
        explanation = []
        if income <= 0.0:
            total_tax = 0.0
            explanation.append('Incomes must be positive.')
            return total_tax, explanation        
        for i, upper in enumerate(income_table[0]):
            if income < upper:
                # Found the right band. Use linear interpolation.
                explanation.append('''\
This household income falls between national average decile %s (which has average gross household income of %.2f, \
and pays %.2f in direct and %.2f in indirect tax), and decile %s (which has average gross \
household income of %.2f, and pays %.2f in direct and %.2f in indirect tax).''' \
                % (i-1, lower,income_table[1][i-1], income_table[2][i-1], \
                   i, upper, income_table[1][i], income_table[2][i]))
                direct_tax = income_table[1][i-1] + (income-lower) * (income_table[1][i]-income_table[1][i-1]) / (upper-lower)
                indirect_tax = income_table[2][i-1] + (income-lower) * (income_table[2][i]-income_table[2][i-1]) / (upper-lower)
                break
            else:
                lower = upper
        else:
            # Above all the bands. Use constant tax rate.
            direct_top_rate = income_table[1][-1] / income_table[0][-1]
            indirect_top_rate = income_table[2][-1] / income_table[0][-1]
            explanation.append('''\
For very high-earning households, in the top half of the top income decile, we assume the fraction of income paid as direct \
tax is %.1f%%, and as indirect tax is %.1f%% - the same as the average for the top income decile.''' % (direct_top_rate*100, indirect_top_rate*100))
            direct_tax = income * direct_top_rate
            indirect_tax = income * indirect_top_rate
        explanation.append('''\
By interpolation, a household with an income of %.2f pays approximately %.2f in direct tax, and %.2f in indirect tax.\
''' % (income, direct_tax, indirect_tax))
        total_tax = direct_tax + indirect_tax
        return total_tax, explanation