from sqlalchemy import Table, Column, ForeignKey
from sqlalchemy.types import *
from sqlalchemy.orm import mapper, relation, backref
import swiss.date

import meta
from base import DomainObject, PublishedDomainObject
from keyvalue import add_keyvalues

class Dataset(PublishedDomainObject):
    def as_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'currency': self.currency,
        }

    def as_big_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'currency': self.currency,
            'metadata': self.metadata,
            'notes': self.notes,
        }

class Entry(PublishedDomainObject):
    # TODO: methods for retrieving 'from' and 'to' account names, and 'time'.

    def as_dict(self):
        dict_ = self.classification_as_dict()
        dict_.update(self.keyvalues_as_dict())
        dict_.update({
            'id': self.id,
            'dataset_id': self.dataset_id,
            'amount': self.amount,
            'currency': self.currency,
        })
        return dict_

    def as_big_dict(self):
        out = self.as_dict()
        cis = dict([(ci.value.key.name, ci.value)
            for ci in self.classificationitems])
        out.update({
            'dataset': self.dataset_.name,
            'notes': self.notes,
        })
        if 'region' in cis:
            out['location'] = cis['region'].name
            out['location_code'] = cis['region'].code
        if 'year' in cis and 'time' not in out:
            out['time'] = cis['year']
        if 'from' in cis:
            # HACK: got to be simpler but just for the moment
            out['from'] = cis['from'].name
            out['from_code'] = cis['from'].code
        if 'to' in cis:
            # HACK: got to be simpler but just for the moment
            out['to'] = cis['to'].name
            out['to_code'] = cis['to'].code
        if 'time' in out:
            try: # ignore bad dates ... (ugly)
                ourdt = swiss.date.parse(out['time']).as_datetime()
                # add Z as solr requires this
                out['time_norm'] = ourdt.isoformat() + 'Z'
            except:
                pass
        return out

    def classification_as_dict(self):
        '''
        Returns all ClassificationItems of this Entry in the form of a
        dict from Key.names to EnumerationValue.codes.
        '''
        return dict([(item.value.key.name, item.value.code)
            for item in self.classificationitems])
    
    def summarise(self):
        '''
        Returns a (unicode, dict) pair describing this Entry. The unicode
        string is a human-readable summary of the key facts about the
        Entry. The dict is from Key.names to EnumerationValues.
        '''
        cis = dict([(ci.value.key.name, ci.value)
            for ci in self.classificationitems])
        def value(name):
            ev = cis[name]
            if not ev:
                return u'unknown'
            elif ev.code == ev.name:
                return ev.code
            else:
                return u'%s (%s)' % (ev.code, ev.name)
        ans = []
        currency = self.currency or self.dataset_.currency or ''
        currency = currency.upper()
        ans.append(u'%s %.2f was paid' % (currency, self.amount))
        if 'from' in cis:
            ans.append(u'\n  from %s' % value('from'))
        if 'to' in cis:
            ans.append(u'\n  to %s' % value('to'))
        if 'time' in cis:
            ans.append(u'\n  in period %s' % value('time'))
        ans.append(u'.')
        ans.append(u'\n\nOther classifiers:')
        for name in sorted(cis.keys()):
            if name not in ['from', 'to', 'time']:
                ans.append(u'\n  %s: %s' % (name, value(name)))
        if False: # self.is_aggregate == u'Y':
            ans.append(u'\nThis is an aggregate formed by summing over all other classifiers.')
        else:
            ans.append(u'\nAll other classifiers are unknown for this entry.')
        ans.append(u' Recorded in data set: %s' % self.dataset_.name)
        return u''.join(ans), cis

table_dataset = Table('dataset', meta.metadata,
    Column('id', Integer, primary_key=True),
    Column('name', UnicodeText(), unique=True),
    Column('long_name', UnicodeText(), unique=True),
    Column('currency', UnicodeText()),
    Column('metadata', UnicodeText(),nullable=True),
    Column('notes', UnicodeText()),
    )

table_entry = Table('entry', meta.metadata,
    Column('id', Integer, primary_key=True),
    Column('dataset_id', Integer, ForeignKey('dataset.id')),
    Column('amount', Float()),
    Column('currency', UnicodeText(),nullable=True),
    Column('notes', UnicodeText()),
    )

mapper(Dataset, table_dataset,
    order_by=table_dataset.c.name
    )

mapper(Entry, table_entry, properties={
        'dataset_': relation(Dataset, backref=backref('entries')),
        },
#    order_by=table_entry.c.timestamp
    )

add_keyvalues(Entry)

