import logging

from pylons import config
from pylons import request, response, session, tmpl_context as c, url, app_globals
from pylons.controllers.util import abort, redirect
from pylons.decorators.cache import beaker_cache

from wdmmg.lib.base import BaseController, render
from wdmmg.controllers.rest import jsonpify
import wdmmg.model as model
import wdmmg.lib.aggregator as aggregator
import wdmmg.lib.calculator as calculator

log = logging.getLogger(__name__)
DEBUG = bool(config.get('debug', False))

try:
    import json
except ImportError:
    import simplejson as json

class ApiController(BaseController):
    def index(self):
        c.rest_url = url(controller='rest', action='index')
        # Construct query strings by hand to keep the parameters in an instructive order.
        c.aggregate_url = url(controller='api', action='aggregate') + \
            '?dataset=%s' % app_globals.default_dataset + \
            '&include-cofog1=07&breakdown-from=yes&breakdown-region=yes'
        c.mytax_url = url(controller='api', action='mytax') + \
            '?income=20000&spending=10000&smoker=yes&driver=yes'
        c.jsonp_url = '&'.join(c.mytax_url.split('&')[:-2] +
          ['callback=myfun'])
        return render('home/api.html')

    # Consider moving _jsonify and _jsonpify to superclass?
    def _jsonify(self, result):
        response.content_type = 'application/json'
        out = json.dumps(result)
        # Note: pylons will automatically convert to relevant charset.
        return unicode(out)

    def _jsonpify(self, result):
        response.content_type = 'text/javascript'
        return u'%s(%s);' % (request.params['callback'],
                json.dumps(result))

    def search(self):
        solrargs = dict(request.params)
        rows = min(1000, request.params.get('rows', 10))
        q = request.params.get('q', '*:*')
        solrargs['q'] = q
        solrargs['rows'] = rows
        solrargs['wt'] = 'json'
        solrargs['sort'] = 'score desc, amount desc'
        if 'callback' in solrargs and not 'json.wrf' in solrargs:
            solrargs['json.wrf'] = solrargs['callback']
        if not 'sort' in solrargs:
            solrargs['sort'] = 'score desc,amount desc'
        query = app_globals.solr.raw_query(**solrargs)
        return query

    @jsonpify
    def aggregate(self):
        # delete the callback param if it exists so that we don't cache on it
        if 'callback' in request.params:
            del params['callback']
        return self._aggregate(
            request.params.keys(),
            values=request.params.values()
            )

    @beaker_cache(type='dbm',
        invalidate_on_startup=DEBUG, # So we can still develop.
        expire=864000, # 10 days.
    )
    def _aggregate(self, keys, values):
        aggregator_params = request.params
        if aggregator_params.get('slice'):
            name_or_id = aggregator_params.get('slice')
        elif aggregator_params.get('dataset'):
            name_or_id = aggregator_params.get('dataset')
        else:
            raise Exception, "Dataset not defined"
        dataset_ = self.get_by_name_or_id(model.Dataset,
            name_or_id=unicode(name_or_id))
        # Retrieve request parameters of the form "verb-key=value"
        include, axes, per, per_time = [], [], [], []
        for param,value in request.params.items():
            if param.startswith('exclude-'):
                pass
            elif param.startswith('include-'):
                key = (model.Session.query(model.Key)
                    .filter_by(name=unicode(param[8:]))
                    ).one() # FIXME: Nicer error message needed.
                include.append((key, value))
            elif param.startswith('breakdown-'):
                key = (model.Session.query(model.Key)
                    .filter_by(name=unicode(param[10:]))
                    ).one() # FIXME: Nicer error message needed.
                axes.append(key) # Value ignored (e.g. "yes").
                # keys for breakdown get added to 'axes'
            elif param.startswith('per-'):
                if value and value!='time':
                    statistic = (model.Session.query(model.Key)
                        .filter_by(name=unicode(param[4:]))
                        ).one() # FIXME: Nicer error message needed.
                    axis = (model.Session.query(model.Key)
                        .filter_by(name=unicode(value))
                        ).one() # FIXME: Nicer error message needed.
                    per.append((axis, statistic))
                else:
                    name = param[4:]
                    assert name in aggregator.time_series, value # FIXME: Nicer error message needed.
                    per_time.append(name)
            # TODO: Other verbs?
            elif param not in ('dataset', 'slice', 'start_date', 'end_date', 'callback'):
                abort(status_code=400,
                  detail='Unknown request parameter: %s'%param)
        results = aggregator.aggregate(
            dataset_,
            include,
            axes,
        )
        for axis, statistic in per:
            results.divide_by_statistic(axis, statistic)
        for statistic_name in per_time:
            results.divide_by_time_statistic(statistic_name)
        ans = {
            'metadata': {
                'dataset': dataset_.name,
                'include': [(k.name, v) for (k, v) in include],
                'dates': [unicode(d) for d in results.dates],
                'axes': results.axes,
                'per': [(a.name, s.name) for a, s in per],
                'per_time': per_time
            },
            'results': results.matrix.items(),
        }
        return ans

    def mytax(self):
        def float_param(name, required=False):
            if name not in request.params:
                if required:
                    abort(status_code=400, detail='parameter %s is missing'%name)
                return None
            ans = request.params[name]
            try:
                return float(ans)
            except ValueError:
                abort(status_code=400, detail='%r is not a number'%ans)
        def bool_param(name, required=False):
            if name not in request.params:
                if required:
                    abort(status_code=400, detail='parameter %s is missing'%name)
                return None
            ans = request.params[name].lower()
            if ans=='yes': return True
            elif ans=='no': return False
            else: abort(status_code=400, detail='%r is not %r or %r'%(ans, 'yes', 'no'))
        tax, explanation = calculator.TaxCalculator2010().total_tax(
            float_param('income', required=True),
            float_param('spending'),
            bool_param('smoker'),
            bool_param('driver')
        )
        result = {'tax': tax, 'explanation': explanation}
        if 'callback' in request.params:
            return self._jsonpify(result)
        else:
            return self._jsonify(result)

