# cli.py
#
# The leaf classes in this module implement Paste Script commands; they
# are runnable with "paster thing", where thing is one of the commands
# listed in the wdmmg section of "paster help".
#
# They are registered in wdmmg's setup.py.
#
# See http://pythonpaste.org/script/developer.html for documentation of
# Paste Script.

import os
import sys

import paste.script.command
import paste.proxy, paste.fixture

class WdmmgCommand(paste.script.command.Command):
    parser = paste.script.command.Command.standard_parser(verbose=True)
    parser.add_option('-c', '--config', dest='config',
            default='development.ini', help='Config file to use (default: development.ini)')
    default_verbosity = 1
    group_name = 'wdmmg'

    def _load_config(self):
        from paste.deploy import appconfig
        from wdmmg.config.environment import load_environment
        if not self.options.config:
            msg = 'No config file supplied'
            raise self.BadCommand(msg)
        self.filename = os.path.abspath(self.options.config)
        conf = appconfig('config:' + self.filename)
        load_environment(conf.global_conf, conf.local_conf)

    def _setup_app(self):
        cmd = paste.script.appinstall.SetupCommand('setup-app') 
        cmd.run([self.filename]) 


class ManageDb(WdmmgCommand):
    '''Perform various tasks on the database.
    
    db create
    db clean
    db rebuild # clean and create
    db init # create and put in default data
    # db upgrade [{version no.}] # Data migrate
    '''
    summary = __doc__.split('\n')[0]
    usage = __doc__
    max_args = None
    min_args = 1

    def command(self):
        self._load_config()
        from wdmmg import model

        cmd = self.args[0]
        if cmd == 'create':
            model.repo.create_db()
        elif cmd == 'init':
            model.repo.init_db()
        elif cmd == 'clean' or cmd == 'drop':
            model.repo.clean_db()
        elif cmd == 'rebuild':
            model.repo.rebuild_db()
        elif cmd == 'upgrade':
            if len(self.args) > 1:
                model.repo.upgrade_db(self.args[1])
            else:
                model.repo.upgrade_db()
        else:
            print 'Command %s not recognized' % cmd
            sys.exit(1)


class Fixtures(WdmmgCommand):
    '''Setup (and teardown) fixtures.

        setup: setup fixtures
        teardown: teardown fixtures
    
    '''
    summary = __doc__.split('\n')[0]
    usage = __doc__
    max_args = None
    min_args = 1

    def command(self):
        self._load_config()
        cmd = self.args[0]
        if cmd == 'setup':
            self.setup()
        elif cmd == 'teardown':
            self.teardown()

    @classmethod
    def setup(self):
        from wdmmg import model
        import pkg_resources, json
        from wdmmgext.load import cofog
        from wdmmgext.load import cra
        model.repo.delete_all()
        model.Session.remove()
        cofog.load_file(pkg_resources.resource_stream('wdmmg', 'tests/COFOG_english_structure_short.txt'))
        cofog_mapper = cra.CofogMapper(json.load(pkg_resources.resource_stream('wdmmg', 'tests/cofog_map_short.json')))
        cra.load_file(pkg_resources.resource_stream('wdmmg', 'tests/cra_2009_db_short.csv'), cofog_mapper)
        cra.load_population(pkg_resources.resource_stream('wdmmg', 'tests/nuts1_population_2006.csv'))
        model.Session.commit()
        model.Session.remove()
        self.dataset_ = (model.Session.query(model.Dataset)
            .filter_by(name=u'cra')
            ).one()
        self.from_ = model.Session.query(model.Key).filter_by(name=u'from').one()
        self.time = model.Session.query(model.Key).filter_by(name=u'time').one()
        self.cap_or_cur = model.Session.query(model.Key).filter_by(name=u'cap_or_cur').one()
        self.pog = model.Session.query(model.Key).filter_by(name=u'pog').one()
        self.region = model.Session.query(model.Key).filter_by(name=u'region').one()
        self.cofog1 = model.Session.query(model.Key).filter_by(name=u'cofog1').one()
        self.cofog2 = model.Session.query(model.Key).filter_by(name=u'cofog2').one()
        self.cofog3 = model.Session.query(model.Key).filter_by(name=u'cofog3').one()
    
    @classmethod
    def teardown(self):
        from wdmmg import model
        model.repo.delete_all()
        model.Session.remove()


class Load(WdmmgCommand):
    '''Load external data into domain model.

        cra: Load CRA data.
        cra2010: Load CRA 2010 data.
        cofog: Load COFOG data.
        barnet: Load Barnet data.
        gla: Load GLA data. (untested)
    '''
    summary = __doc__.split('\n')[0]
    usage = __doc__
    max_args = None
    min_args = 1

    def command(self):
        self._load_config()
        cmd = self.args[0]
        import pkg_resources
        for entry_point in pkg_resources.iter_entry_points('wdmmg.load'):
            if entry_point.name == cmd:
                loader_method = entry_point.load()
        if loader_method:
            loader_method(*self.args[1:])
        else:
            print 'Loader not recognized'

class SiteTest(WdmmgCommand):
    '''Perform simple consistency tests on the UI.
       Checks the CRA dataset, key and key-values exist in the API. 
       Checks the search page exists (though not that it returns entries):
       expect this to fail if you don't have Solr installed.
       Checks aggregate data for some errors that break Flash.
       Usage: paster sitetest <url>
    '''
    summary = __doc__.split('\n')[0]
    usage = __doc__
    max_args = 1
    min_args = 1
    def command(self):
        self._load_config()
        self.wui_address = self.args[0]
        if not self.wui_address.startswith('http://'):
            self.wui_address = 'http://' + self.wui_address    
        wsgiapp = paste.proxy.make_proxy({}, self.wui_address)
        self.app = paste.fixture.TestApp(wsgiapp)
        def check_page(path, required_contents, present=True, status=200):
            print "* Checking page '%s%s'" % (self.wui_address, path)
            res = self.app.get(path, status=status)
            if type(required_contents) is type(()):
                for required in required_contents:
                    print '    ...checking for %r' % required
                    assert required in res, res
            else:
                assert required_contents in res, res
            return res
        check_page('/', 'Welcome')
        check_page('/entry', 'Entries')
        check_page('/key', 'keys')
        check_page('/key/time', 'time')
        check_page('/api/rest/dataset/cra', 'cra')
        check_page('/api/rest/key/cofog1', 'Health')
        check_page('/api/rest/key/cofog1/value/05', 'The breakdown of environmental protection')
        check_page('/api/rest/key/cofog1/value/05', 'The breakdown of environmental protection')
        check_page('/search?q=cra', 'results')
        check_page('/api/search?q=cra', 'numFound')
                            
class Solr(WdmmgCommand):
    '''Use solr index.

        load {dataset}: load data for dataset into solr.
        delete {dataset}: delete data for dataset from solr.
    '''
    summary = __doc__.split('\n')[0]
    usage = __doc__
    max_args = None
    min_args = 2
    
    def command(self):
        self._load_config()
        cmd = self.args[0]
        import wdmmg.lib.solrhelp
        if cmd == 'load':
            wdmmg.lib.solrhelp.build_index(self.args[1])
        elif cmd == 'delete':
            wdmmg.lib.solrhelp.drop_index(self.args[1])
        else:
            print 'Action not recognized'

