import re

from wdmmg.tests import *
import wdmmg.lib.solrhelp


class TestApiController(TestController):
    @classmethod
    def setup_class(self):
        Fixtures.setup()
    
    @classmethod
    def teardown_class(self):
        Fixtures.teardown()

    def test_index(self):
        response = str(self.app.get(url(controller='api', action='index')))
        assert '''the following requests:''' in response, response
        assert re.search(r'api/aggregate\?.*dataset=', response), response
        assert re.search(r'api/aggregate\?.*include-', response), response
        assert re.search(r'api/aggregate\?.*breakdown-', response), response

    def test_aggregate(self):
        response = self.app.get(url(controller='api', action='aggregate',
            dataset='cra'))
        assert '"metadata": {' in response, response
        assert '"dataset": "cra"' in response, response
        assert '"include": []' in response, response
        assert '"dates": ["' in response, response
        assert '"axes": []' in response, response
        assert '"results": [[' in response

    def test_aggregate_with_breakdown(self):
        u = url(controller='api', action='aggregate', **{
            'dataset': 'cra',
            'breakdown-region': 'yes',
        })
        print u
        response = self.app.get(u)
        assert '"axes": ["region"]' in response, response
        assert '"ENGLAND_London"' in response, response
       
    def test_aggregate_respects_breakdown_key_ordering(self):
        u = url(controller='api', action='aggregate', dataset='cra')
        breakdowns = ['cofog1', 'region']
        # explicitly append to ensure ordering
        u += '&' + '&'.join(['breakdown-%s=yes' % x for x in breakdowns ])
        print u
        response = self.app.get(u)
        out = json.loads(response.body)
        outorder = out['metadata']['axes']
        assert outorder == breakdowns, outorder

        u = url(controller='api', action='aggregate', dataset='cra')
        breakdowns = ['region', 'cofog1']
        # explicitly append to ensure ordering
        u += '&' + '&'.join(['breakdown-%s=yes' % x for x in breakdowns ])
        print u
        response = self.app.get(u)
        out = json.loads(response.body)
        outorder = out['metadata']['axes']
        assert outorder == breakdowns, outorder

    def test_jsonp_aggregate(self):
        # Copied from test_aggregate_with_breakdown.
        callback = randomjsonpcallback()
        u = url(controller='api',
            callback=callback, action='aggregate', **{
            'dataset': 'cra',
            'breakdown-region': 'yes',
        })
        print u
        response = self.app.get(u)
        print response, str(response)
        assert '"axes": ["region"]' in response, response
        assert '"ENGLAND_London"' in response, response
        assert valid_jsonp(response, callback)

    def test_aggregate_with_per(self):
        u = url(controller='api', action='aggregate', **{
            'dataset': 'cra',
            'breakdown-region': 'yes',
            'per-population2006': 'region'
        })
        print u
        response = self.app.get(u)
        assert '"axes": ["region"]' in response, response
        assert '"ENGLAND_London"' in response, response
        assert '0.1' in response, response

    def test_aggregate_with_per_time(self):
        u = url(controller='api', action='aggregate', **{
            'dataset': 'cra',
            'per-gdp_deflator2006': ''
        })
        print u
        response = self.app.get(u)
        assert '"axes": []' in response, response
        assert '"2006"' in response, response
        assert '18445770.0' in response, response

    def test_mytax(self):
        u = url(controller='api', action='mytax', income=20000)
        print u
        response = self.app.get(u)
        assert '"tax": ' in response, response
        assert '"explanation": ' in response, response

    def test_jsonp_mytax(self):
        # Copied from test_mytax.
        callback = randomjsonpcallback()
        u = url(controller='api', action='mytax', income=20000,
          callback=callback)
        print u
        response = self.app.get(u)
        assert '"tax": ' in response, response
        assert '"explanation": ' in response, response
        assert valid_jsonp(response, callback)

def randomjsonpcallback(prefix='cb'):
    """Generate a random identifier suitable, beginning with *prefix*,
    for using as a JSONP callback name."""

    import random
    import string
    return prefix + ''.join(random.choice(string.letters) for
      _ in range(6))

def valid_jsonp(response, callback):
    """True if *response* is valid JSONP using *callback* as the
    callback name.  Currently does not completely validate
    everything."""

    return (
        ((callback + '(') in response, response) and
        (str(response)[-2:] == ');' or str(response)[-1] == ')')
           )


from wdmmg.controllers.api import json
class TestApiSearch(TestController):
    @classmethod
    def setup_class(self):
        Fixtures.setup()
        # We have to pass in 'wdmmg.lib.solrhelp.stub' explicitly, because
        # pylons.app_globals does not exist outside a web request.
        self.solr = wdmmg.lib.solrhelp.solr_connection()
        self.solr.delete_query('*:*')
        wdmmg.lib.solrhelp.build_index(dataset_name=Fixtures.dataset_.name,
                solr=self.solr)
                
    @classmethod
    def teardown_class(self):
        Fixtures.teardown()

    def test_search_01_no_query(self):
        response = self.app.get(url(controller='api', action='search'))
        out = json.loads(str(response.body))['response']
        assert out['numFound'] == 36, out['numFound']
        assert out['docs'][0]['dataset'] == 'cra', out

    def test_search_02_query(self):
        response = self.app.get(url(controller='api', action='search',
            q='Children'))
        out = json.loads(str(response.body))['response']
        assert out['numFound'] == 7, out['numFound']
        exp_entity = 'Department for Children, Schools and Families'
        assert out['docs'][0]['from'] == exp_entity, out['docs'][0]

    def test_search_03_jsonpify(self):
        callback = 'mycallback'
        response = self.app.get(url(controller='api', action='search',
            q='children', callback=callback))
        assert response.body.startswith('%s({"responseHeader"' % callback), response.body

