import os
import sys
import datetime

import rfc3339

try:
    from jinja2 import Environment, FileSystemLoader, ChoiceLoader
    from jinja2 import environmentfilter, contextfilter, Markup

    @contextfilter
    def renderstring(context, value):
        '''
        Render the passed string. It is similar to the tag rendertemplate,
        except it uses the passed string as the template.

        Example:
        The template 'Hello {{ string_template|renderstring }}!';
        Called with the following context:
            dict(string_template='{{ foo }} world',
                 foo='crazy')
        Renders to:
        'Hello crazy world!'
        '''
        if value:
            env = context.environment
            result = env.from_string(value).render(context.get_all())
            if env.autoescape:
                result = Markup(result)
            return result
        else:
            return ''

    def rfc3339_(value):
        return rfc3339.rfc3339(value)

    def decode(value):
        if value:
            return value.encode('ascii', 'xmlcharrefreplace')
        else:
            return ''

except ImportError:
    raise SystemExit('Please install Jinja 2 (http://jinja.pocoo.org/2/)'
                     ' to use Weblog')

def jinja_environment(source_dir):
    """
    Build the Jinja environment. Setup all template loaders.
    """
    TEMPLATE_DIR = 'templates'
    fs_loader = FileSystemLoader(os.path.join(source_dir, TEMPLATE_DIR))
    fs_app_loader = FileSystemLoader(os.path.join(sys.path[0], 'weblog',
                                                  TEMPLATE_DIR))
    # if setuptools is present use the loader else fake it.
    try:
        import pkg_resources
        from jinja import PackageLoader
    except ImportError:
        pkg_loader = FileSystemLoader(os.path.join(os.path.dirname(__file__),
                                                   TEMPLATE_DIR))
    else:
        pkg_loader = PackageLoader('weblog', TEMPLATE_DIR)
    choice_loader = ChoiceLoader([fs_loader, fs_app_loader, pkg_loader])
    env = Environment(loader=choice_loader, trim_blocks=True)
    env.filters['renderstring'] = renderstring
    env.filters['rfc3339'] = rfc3339_
    env.filters['decode'] = decode
    return env
