'Methods for formatting and parsing friendly timestamps'
import re
import datetime


indexByWeekday = {
    'mon': 0, 'monday': 0,
    'tue': 1, 'tuesday': 1,
    'wed': 2, 'wednesday': 2,
    'thu': 3, 'thursday': 3,
    'fri': 4, 'friday': 4,
    'sat': 5, 'saturday': 5,
    'sun': 6, 'sunday': 6,
}
dateTemplates = '%m/%d/%Y', '%m/%d/%y', '%m/%d'
timeTemplates = '%I%p', '%I:%M%p'
pattern_offset = re.compile(r'([+-]\d\d)(\d\d) UTC')


class WhenIO(object):
    'Convenience class for formatting and parsing friendly timestamps'
    
    # Constructor

    def __init__(self, offsetMinutes=0, localToday=None):
        """
        Set user-specific parameters.
        offsetMinutes  Set offset as returned by Javascript's Date.getTimezoneOffset()
        localToday     Set reference date to parse special terms like "Today"
        """
        # Set
        self.offsetMinutes = offsetMinutes
        self.localToday = localToday.date() if localToday else self.to_local(datetime.datetime.utcnow()).date()
        self.localTomorrow = self.localToday + datetime.timedelta(days=1)
        self.localYesterday = self.localToday + datetime.timedelta(days=-1)

    # Format

    def format(self, whens, dateTemplate=dateTemplates[0], dateTemplate_='', fromUTC=True, separator=' '):
        """
        Format whens into strings.
        whens          A timestamp or list of timestamps
        dateTemplate   Date template to use when the date is more than seven days from today
        dateTemplate_  Date template to use when the date is less than seven days from today
        fromUTC=True   Convert whens to local time before formatting
        fromUTC=False  Format whens without conversion
        """
        # Convert
        if not isinstance(whens, list):
            whens = [whens]
        if fromUTC:
            whens = map(self.to_local, whens)
        # Initialize
        strings = []
        previousDate = None
        # For each when,
        for when in whens:
            # Ignore null
            if when == None: 
                continue
            # If the when matches the previousDate,
            if when.date() == previousDate:
                # Only format time
                string = self.format_time(when)
            # Otherwise,
            else:
                # Format
                string = '%s %s' % (self.format_date(when, dateTemplate, dateTemplate_), self.format_time(when))
            # Append
            strings.append(string)
            # Prepare for next iteration
            previousDate = when.date()
        # Return
        return separator.join(strings)

    def format_date(self, date, dateTemplate=dateTemplates[0], dateTemplate_=''):
        """
        Format date into string.
        dateTemplate   Use this template if the date is one week from localToday
        dateTemplate_  Use this template otherwise
        """
        # Convert
        if isinstance(date, datetime.datetime): 
            date = date.date()
        dateString = date.strftime(dateTemplate_) if dateTemplate_ else ''
        # Format special
        if date == self.localToday: 
            return 'Today' + dateString
        elif date == self.localTomorrow: 
            return 'Tomorrow' + dateString
        elif date == self.localYesterday: 
            return 'Yesterday' + dateString
        # Format weekday
        differenceInDays = (date - self.localToday).days
        if differenceInDays <= 7 and differenceInDays > 0:
            return date.strftime('%A') + dateString
        # Return
        return date.strftime(dateTemplate)

    def format_time(self, time):
        'Format time into string'
        # Convert
        if isinstance(time, datetime.datetime): 
            time = time.time()
        # Format
        hour = time.hour % 12
        if not hour: 
            hour = 12
        hour = str(hour)
        if time.minute == 0: 
            return hour + time.strftime('%p').lower()
        # Return
        return hour + time.strftime(':%M%p').lower()

    def format_offset(self):
        'Format timezone offset'
        return format_offset(self.offsetMinutes)

    # Parse

    def parse(self, whensString, toUTC=True):
        """
        Parse whens from strings.
        whensString  A string of timestamps
        toUTC=True   Convert whens to UTC after parsing
        toUTC=False  Parse whens without conversion
        """
        # Initialize
        whens, terms = [], []
        # For each term,
        for term in whensString.split():
            # Try to parse the term as a date
            date = self.parse_date(term)
            # If the term is a date,
            if date != None: 
                if date not in whens:
                    whens.append(date)
                continue
            # Try to parse the term as a time
            time = self.parse_time(term)
            # If the term is a time,
            if time != None: 
                # Load
                oldWhen = whens[-1] if whens else None
                newWhen = self.combine_date_time(oldWhen, time)
                # If oldWhen already has a time or we have no whens, append newWhen
                if isinstance(oldWhen, datetime.datetime) or not whens:
                    whens.append(newWhen)
                # If oldWhen did not have a time, replace oldWhen
                else:
                    whens[-1] = newWhen
                continue
            # If it is neither a date nor a time, save it
            if term not in terms:
                terms.append(term)
        # Make sure every when has a time
        for whenIndex, when in enumerate(whens):
            if not isinstance(when, datetime.datetime):
                whens[whenIndex] = self.combine_date_time(when, whenTime=None)
        # Convert
        if toUTC:
            whens = map(self.from_local, whens)
        # Return
        return sorted(whens), terms

    def parse_date(self, dateString):
        'Parse date from a string'
        # Prepare string
        dateString = dateString.strip().lower()
        # Look for special terms
        if dateString in ('today', 'tod'):
            return self.localToday
        elif dateString in ('tomorrow', 'tom'):
            return self.localTomorrow
        elif dateString in ('yesterday', 'yes'):
            return self.localYesterday
        elif dateString in indexByWeekday:
            difference = indexByWeekday[dateString] - self.localToday.weekday()
            if difference <= 0: 
                difference += 7
            return self.localToday + datetime.timedelta(days=difference)
        # Parse date
        for template in dateTemplates:
            try: 
                date = datetime.datetime.strptime(dateString, template).date()
            except (ValueError, TypeError):
                pass
            else:
                if date.year == 1900:
                    date = date.replace(year=self.localToday.year)
                return date

    def parse_time(self, timeString):
        'Parse time from a string'
        # Prepare string
        timeString = timeString.strip().lower()
        # Parse time
        for template in timeTemplates:
            try: 
                time = datetime.datetime.strptime(timeString, template).time()
            except (ValueError, TypeError):
                pass
            else: 
                return time

    # Helpers

    def combine_date_time(self, whenDate, whenTime=None):
        'Create when from date and time, assuming where necessary'
        # If both terms are present, combine them            
        if whenTime != None and whenDate: 
            return datetime.datetime.combine(whenDate, whenTime)
        # If only the time is present,
        if whenTime != None: 
            return datetime.datetime.combine(self.localToday, whenTime)
        # If only the date is present,
        if whenDate: 
            return datetime.datetime.combine(whenDate, datetime.time(0, 0))

    def from_local(self, when):
        'Convert whenLocal into UTC'
        if when:
            return when + datetime.timedelta(minutes=self.offsetMinutes)

    def to_local(self, when):
        'Convert UTC into whenLocal'
        if when: 
            return when - datetime.timedelta(minutes=self.offsetMinutes)


def format_offset(offsetMinutes):
    'Format timezone offset'
    hourCount = offsetMinutes / 60.
    hourQuotient = int(hourCount)
    hourRemainder = abs(hourCount - hourQuotient)
    return '%+03d%02d UTC' % (-1 * hourCount, hourRemainder * 60)


def parse_offset(text):
    'Parse timezone offset'
    match = pattern_offset.search(text)
    if match:
        x, y = match.groups()
        x = int(x)
        y = int(y)
        offsetMinutes = (1 if x < 0 else -1) * (abs(x) * 60 + y)
        text = pattern_offset.sub('', text)
    else:
        offsetMinutes = None
    return offsetMinutes, text


def format_interval(rdelta, precision=2):
    'Format a relativedelta object rounded to the given precision'
    units = ['years', 'months', 'days', 'hours', 'minutes', 'seconds']
    maxes = [12, 30, 24, 60, 60, 1000000]
    packs = []
    precisionIndex = 0
    for unit in units:
        value = getattr(rdelta, unit)
        # If we have an empty value,
        if not value:
            # If we have skipped a unit, break
            if packs:
                break
            # If we have no terms yet, continue
            continue
        packs.append([value, unit])
        precisionIndex += 1
        # If we are at the requested precision,
        if precisionIndex >= precision:
            unitIndex = units.index(unit)
            if unitIndex < len(units):
                # Look at value of the next unit and round up if necessary
                valueNext = getattr(rdelta, units[unitIndex + 1])
                if valueNext > maxes[unitIndex] / 2:
                    packs[-1][0] += 1
            break
    # Format terms with appropriate plurality
    return ' '.join('%i %s' % (value, unit if abs(value) != 1 else unit[:-1]) for value, unit in packs)


def parse_interval(text):
    'Parse a relativedelta object from a string'
    from dateutil.relativedelta import relativedelta
    units = ['years', 'months', 'weeks', 'days', 'hours', 'minutes', 'seconds']
    terms = text.split()
    valueByUnit = {}
    for termIndex, term in enumerate(terms[1:], 1):
        if not term.endswith('s'):
            term += 's'
        if term in units:
            try:
                value = float(terms[termIndex - 1])
            except ValueError:
                continue
            valueByUnit[term] = value
    return relativedelta(**valueByUnit)
