"""Buildout recipe to merge manual and generated sections of config files.
"""
# Copyright (c) 2014, White Horse Technology Consulting Inc.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import errno
import logging
import os
import re
from shutil import copy

import zc.buildout
from zc.buildout.buildout import Options

DEFAULT_START_MARKER = '# BEGIN - Generated by: ' 
DEFAULT_END_MARKER = '# END - Generated by: ' 
DEFAULT_COMMENT = (
    "# DO NOT EDIT: Text between these lines generated automatically "
    "by buildout"
    )

def _query_bool(options, name, default=None):
    """Replace query_bool missing in buildout2
    """
    if hasattr(options, 'query_bool'):
        return options.query_bool(name, default=default)
    else:
        from zc.buildout.buildout import bool_option
        if default is not None:
            value = options.setdefault(name, default=default)
        else:
            value = options.get(name)
            if value is None:
                return value
        return bool_option(options, name)

def _find_sections(contents, start_marker, end_marker, strict, logger=None):
    """Return list of tuples marking the locations of existing sections.
    
    Location is marked by first (zero-based) and last line numbers. These
    need to be modified when using slice notation.
    
    We pass parameters as this may be called by both install and 
    uninstall.
    """
    found = False
    start_index = 0
    line_counter = 0
    sections = []
    for line in contents:
        line = line.rstrip('\n')
        if not found:
            if line.strip().startswith(start_marker):
                found = True
                start_index = line_counter
        else:
            if line.strip().startswith(end_marker):
                found = False
                sections.append((start_index, line_counter))
        line_counter += 1

    if found:
        msg = "End of file reached without finding an end marker"
        if strict:
            raise zc.buildout.UserError(msg)
        else:
            sections.append(
                (start_index, line_counter and line_counter - 1 or 0)
                )
            if logger:
                logger.warning(msg)
            else:
                print msg
    
    return sections

def _remove_sections(sections, contents):
    """Given a list of section markers, remove them from the contents.
    
    If section markers is empty the original contents are returned.
    """
    if not sections:
        return contents

    start_index = 0
    new_contents = []
    for section in sections:
        new_contents.extend(contents[start_index:section[0]])
        start_index = section[1] + 1
        
    new_contents.extend(contents[start_index:])
    return new_contents

class Recipe:
    def __init__(self, buildout, name, options):
        """Merge buildout generated sections of a manually edited config file.
        
        This tool will read a file, looking for marker text to indicate the 
        beginning and end of a section. If it finds it, the marker text and 
        the content between it will be replace by the generated contents. If 
        not, the marker text and contents will be added to the end of the file.
        
        This lets us add and update buildout-generated sections within a file
        that may have other, manually-managed sections.
        
        Options:

        """
        self.buildout = buildout
        self.name = name
        self.options = options
        self.logger = logging.getLogger(self.name)
        self.msg = None
        self.verbosity = int(buildout['buildout'].get('verbosity', 0))

        self.comment = options.get('comment', DEFAULT_COMMENT).strip()
        self.insert_after = options.get('insert-after', '').strip()
        
        # Process these so the defaults are stored for uninstall 
        self.strict = _query_bool(self.options, 'strict', default='false')
        self.create = _query_bool(self.options, 'create', default='true')
        self.uninstall = _query_bool(self.options, 'uninstall', default='true')
        self.allow_empty_section = _query_bool(
            self.options, 
            'allow-empty-section', 
            'false'
            )
        
        if (
                'section' not in options and 
                'section-file' not in options and 
                'section-template' not in options
            ):
                raise zc.buildout.UserError(
                    "No section contents, file or template specified."
                    )

        if (
            ('section' in options and 'section-file' in options) or
            ('section' in options and 'section-template' in options) or
            ('section-file' in options and 'section-template' in options)
            ):
                raise zc.buildout.UserError(
                    "Too many section options specified."
                    )

        self.section = options.get("section", ''.strip())
        self.section_file = options.get("section-file", '').strip()
        self.section_template = options.get("section-template", '').strip()
        
        if 'section-file' in options and not self.section_file:
            raise zc.buildout.UserError("No section file defined.")

        if 'section-template' in options and not self.section_template:
            raise zc.buildout.UserError("No section template defined.")
        
        self.target = options.get("target", '').strip()
        if not self.target:
            raise zc.buildout.UserError(
                "No configuration file (target) specified."
                )

        self.backup = _query_bool(self.options, 'backup', 'true')

        # Store the default markers in options so we have them for uninstall
        options['start-marker'] = options.get(
            'start-marker',
            DEFAULT_START_MARKER + self.name
            )
        self.start_marker = options['start-marker']
        if not self.start_marker:
            raise zc.buildout.UserError(
                "You must define a starting marker line (or omit to use"
                "the default)."
                )

        options['end-marker'] = options.get(
            'end-marker',
            DEFAULT_END_MARKER + self.name
            )
        self.end_marker = options['end-marker']
        if not self.end_marker:
            raise zc.buildout.UserError(
                "You must define an ending marker line (or omit to use"
                "the default)."
                )

        self.delete_file = _query_bool(
            self.options, 
            'delete-file', 
            default='false'
            )
        if self.delete_file:
            if not self.section_file:
                if self.verbosity:
                    self.logger.info(
                        (
                            "Ignoring delete-file because section-file "
                            "is not defined."
                            )
                        )

        # Generate our section now, so that if there's a problem, we exit
        # before uninstall is called to undo prior modifications
        self.section_contents = self._generateSection()

    def _generateSection(self):
        section = []
        if self.section:
            section.extend(self.section.splitlines())
        elif self.section_file:
            section_file = open(self.section_file, 'r')
            for line in section_file:
                section.append(line.rstrip('\n'))
            section_file.close()

            if self.delete_file:
                try:
                    os.remove(self.section_file)
                except(IOError, OSError), err:
                    msg = "Unable to remove section file: %s" % (
                        self.section_file
                        )
                    if self.strict:
                        raise zc.buildout.UserError(msg)
                    else:
                        self.logger.warning(msg, exc_info = True)
                except:
                    raise
                else:
                    if self.verbosity:
                        self.logger.info(
                            "Removed section file: %s" % self.section_file
                            )
        elif self.section_template:
            from collective.recipe.template import Recipe as Template
            # Copying an Options instance returns a dict, so work around that
            options = self.buildout[self.section_template]
            options = Options(self.buildout, options.name, options._raw)
            if 'output' not in options:
                # collective.recipe.template requires an output file
                options['output'] = 'foo'
            template = Template(self.buildout, self.name, options)
            section.extend(template.result.strip().splitlines())

        if not section and not self.allow_empty_section:
            raise zc.buildout.UserError(
                "The section you defined has no contents. If you wish to "
                "insert an empty section, set allow-empty-section to 'true'."
                )
        
        if self.comment:
            section.insert(0, self.comment)
        section.insert(0, self.start_marker)
        section.append(self.end_marker)

        return section

    def install(self, update=False):
        """Install our part.
        """

        contents = []
        try:
            source = open(self.target, "r")
            contents = source.read().splitlines()
            source.close()
        except (IOError, OSError), err:
            if err.errno != errno.ENOENT:
                raise

            if self.create:
                exists = False
            else:
                raise zc.buildout.UserError(
                    "Target file %s does not exist and create is "
                    "set to False."
                    )
        else:
            exists = True

        sections = None
        if exists:
            if self.backup:
                backup = self.target + '.BK0'
                copy(self.target, backup)
                self.logger.info(
                    "Install/Update: %s backed up to %s" % (self.target, backup)
                    )

        # We should only need to remove existing sections on update, 
        # but a little paranoia can be a good thing. 
        sections = _find_sections(
            contents, 
            self.start_marker, 
            self.end_marker, 
            self.strict,
            self.logger
            )        
        contents = _remove_sections(sections, contents)

        # Look for an insertion point, otherwise append
        insert_index = len(contents)
        if self.insert_after:
            pattern = re.compile(self.insert_after)
            for index, line in enumerate(contents):
                match = pattern.search(line)
                if match:
                    insert_index = index + 1
                    break;
        contents[insert_index:insert_index] = self.section_contents

        target = open(self.target, 'w')
        for line in contents:
            target.write("%s\n" % line)
        target.close()

        if not exists and self.verbosity:
            self.logger.info(
                "Target file %s did not exist and has been created"
                )
        return []

    def update(self):
        return self.install(update=True)

def uninstall(name, options):
    """Remove the defined section from the target file."""
    if not options.get('uninstall'):
        return

    strict = _query_bool(options, 'strict')
    try:
        source = open(options['target'], "r")
        contents = source.read().splitlines()
        source.close()    
    except (IOError, OSError), err:
        if err.errno != errno.ENOENT:
            raise

        msg = "Target file %s does not exist for uninstall."
        if strict:
            raise zc.buildout.UserError(msg)
        else: 
            print msg
            return

    sections = _find_sections(
        contents, 
        options.get(
            'start-marker',
            DEFAULT_START_MARKER + name
            ),
        options.get(
            'end-marker',
            DEFAULT_END_MARKER + name
            ), 
        strict,
        )
    contents = _remove_sections(sections, contents)
    
    if options.get('backup'):
        backup = options['target'] + '.BK1'
        copy(options['target'], backup)
        print "Uninstall: %s backed up to %s" % (
            options['target'], 
            backup
            )
    
    if contents:
        target = open(options['target'], 'w')
        for line in contents:
            target.write("%s\n" % line)
        target.close()
    else:
        os.remove(target)

 