# -*- coding: ascii -*-
#
# Copyright 2006-2012 Andr\xe9 Malo or his licensors, as applicable
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Commandline Handling
====================

This modules handles commandline parsing and initial bootstrapping.
"""
__author__ = u"Andr\xe9 Malo"
__docformat__ = "restructuredtext en"

import optparse as _optparse
import os as _os
import sys as _sys

from wtf import Error

class CommandlineError(Error):
    """ Error on command line """


def terminal_width():
    """
    Return terminal width if determined, None otherwise

    :return: The width
    :rtype: ``int``
    """
    fd = None
    def isatty(fd):
        """ Determine if the fd is a TTY """
        try:
            _isatty = bool(_os.isatty(fd))
        except AttributeError:
            _isatty = False
        return _isatty

    for fp in (_sys.stdout, _sys.stdin):
        try:
            _fd = fp.fileno()
        except (AttributeError, ValueError):
            continue
        else:
            if isatty(_fd):
                fd = _fd
                break

    if fd is not None:
        try:
            import fcntl, struct, termios

            # struct winsize { /* on linux in asm/termios.h */
            #     unsigned short ws_row;
            #     unsigned short ws_col;
            #     unsigned short ws_xpixel;
            #     unsigned short ws_ypixel;
            # }
            return struct.unpack("4H", fcntl.ioctl(
                fd, termios.TIOCGWINSZ, struct.pack("4H", 0, 0, 0, 0)
            ))[1]

        except (SystemExit, KeyboardInterrupt):
            raise

        except:
            # don't even ignore
            pass

    return None


class _OptionParser(_optparse.OptionParser): # old style class
    """ Extended option parser for better behaviour """

    def __init__(self, **kwargs):
        """ Initialization """
        kwargs.setdefault('prog', "<prog>")
        kwargs.setdefault('usage', "1")
        kwargs.setdefault('version', "2")
        _optparse.OptionParser.__init__(self, **kwargs)

    def parse_args(self, *args, **kwargs):
        """
        Parse the argument list provided by ``sys.argv``

        :Parameters:
         - `args`: Additional arguments for the parser
         - `kwargs`: Additional arguments for the parser

        :Types:
         - `args`: ``tuple``
         - `kwargs`: ``dict``

        :return: The ``OptionContainer`` instance and the positional args
                 (``(optparse.OptionContainer, list)``)
        :rtype: ``tuple``

        :Exceptions:
         - `CommandlineError`: The argument list was invalid
        """
        # pylint: disable = W0221

        self.prog = _os.path.basename(_sys.argv[0])
        return _optparse.OptionParser.parse_args(self,
            _sys.argv[1:], *args, **kwargs)

    def error(self, msg):
        """
        Raise an exception instead of calling ``sys.exit``

        :Parameters:
         - `msg`: The error message

        :Types:
         - `msg`: ``str``

        :Exceptions:
         - `CommandlineError`: command line error
        """
        raise CommandlineError(str(msg))

    def get_version(self):
        """
        Return the version string

        :return: The version string
        :rtype: ``str``
        """
        from wtf import version
        return str(version)

    def get_usage(self):
        """
        Return the usage string

        :return: The usage string
        :rtype: ``str``
        """
        return "Usage: %s <options> start|stop|logrotate\n" % \
            self.get_prog_name()

    def get_prog_name(self):
        """
        Returns the program name

        :return: The program name
        :rtype: ``str``
        """
        return self.prog
    _get_prog_name = get_prog_name # 2.3.4 <= python < 2.4.0

    def format_help(self, formatter=None):
        """
        Return the formatted help string

        The string consists of the normal option help generated by
        the optparse module and a short description of the old style
        options. All text is tried to be wrapped to fit into the
        current terminal width.

        :Parameters:
         - `formatter`: unused

        :Types:
         - `formatter`: any

        :return: The formatted help string
        :rtype: ``str``
        """
        # determine possible with
        width = (terminal_width() or 80) - 1
        optionhelp = None
        while optionhelp is None:
            formatter = _optparse.IndentedHelpFormatter(width=width)
            try:
                optionhelp = \
                    _optparse.OptionParser.format_help(self, formatter)
            except ValueError:
                # terminal too small
                if width < 79:
                    width = 79
                else:
                    width += 10
        return optionhelp


def parse(*opts):
    """
    Parse command line

    :Parameters:
     - `opts`: Option spec (WRITEME: need more info here :-)
       (``({key: value, ...}, ...)``)
       Special keys are '' and 'ENV'.

    :Types:
     - `opts`: ``tuple``

    :return: The option container and the fixed args
             (``(optparse.OptionContainer, list)``)
    :rtype: ``tuple``
    """
    optlist = []
    for opt in opts:
        opt = opt.copy()
        names = opt.pop('')
        if 'ENV' in opt:
            defaultval = _os.environ.get(opt.pop('ENV'))
            if defaultval is not None:
                opt['default'] = defaultval
        optlist.append(_optparse.make_option(*names, **opt))
    parser = _OptionParser(option_list=optlist)
    return parser.parse_args()


def run():
    """ Main command line runner """
    opts, args = parse(
        {'': ['-c', '--config'], 'action': 'store', 'type': 'string',
            'ENV': 'WTFRC',
            'help': 'Configuration file, overrides the WTFRC variable'},
        {'': ['--dump-config'], 'action': 'store_true',
            'help': 'Load config, dump it to STDOUT and exit'},
        {'': ['--profile'], 'action': 'store', 'type': 'string',
            'metavar': 'FILENAME',
            'help': 'Run with (c)profiler, output to FILENAME'},
        {'': ['--keep-descriptors'], 'action': 'store_true',
            'help': "Don't close all file descriptors during startup "
                    "(useful for debugging)"},
        {'': ['-l', '--listen'], 'action': 'append',
            'help': 'Override the listen configuration in config file '
                    '(Can be used more than once)'},
        {'': ['--checkinterval'], 'action': 'store', 'type': 'int',
            'help': 'Set the number of byte code instructions between thread '
                    'switches (sys.setcheckinterval())'},
        {'': ['--max-descriptors'], 'action': 'store', 'type': 'int',
            'help': 'Set the maximum number of file descriptors '
                    '(ulimit -n)'},
        {'': ['-m', '--workermodel'], 'action': 'store',
            'help': 'Override the workermodel configuration in config file '},
    )
    args = [_sys.argv[0]] + list(args)
    if not opts.config:
        raise CommandlineError("No configuration file given")

    from wtf import init as _init
    config = _init.config(opts.config, opts=opts, dump=opts.dump_config)

    def start():
        """ Starter """
        from wtf import opi as _opi
        try:
            _opi.factory(config, opts, args).work()
        except _opi.OPIError, e:
            msg = str(e)
            if msg:
                print >> _sys.stderr, msg
            _sys.exit(e.exit_code)

    if opts.profile:
        try:
            import cProfile as _profile
        except ImportError:
            import profile as _profile
        _profile.runctx("start()", globals(), locals(), opts.profile)

    else:
        start()
