#!/usr/bin/env/python
# wyckedsceptre_test.py
"""
@package wyckedsceptre_test
Module based on Python's unittest to test each of the classes
created in the wyckedsceptre module, as well as the constants defined
in wyckedsceptre_constants.py

project home: http://code.google.com/p/wyckedsceptre/
Last Changed Date: $LastChangedDate: 2010-10-26 22:37:34 -0500 (Tue, 26 Oct 2$
Revision         : $Rev: 36 $
"""
import unittest
from wyckedsceptre import Die

## Constant lists to be used for testing
CLASS_NAMES = ['cleric', 'fighter', 'thief', 'dwarf', 'elf', 'halfling',
                'magicuser']

SAVING_THROW_TYPES = ['DEATHRAY-POISON', 'MAGICWANDS', 'PARALYSIS-TURNSTONE',
                      'DRAGONBREATH', 'RODS-STAVES-SPELLS']

ABILITY_NAMES = ['STR', 'INT', 'WIS', 'DEX', 'CON', 'CHA']


class TestAbilityScoreOrder(unittest.TestCase):
    """
    unittest TestCase to insure that the constant ABIL_SCORE_ORDER is valid
    and properly filled.  Each class must be listed once, and no others.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre_constants import ABIL_SCORE_ORDER
        self.abilities_list = ABIL_SCORE_ORDER

    def test_data_type(self):
        """
        verifies that each class is a dictionary in the ability scores
        """
        self.assertTrue(isinstance(self.abilities_list, dict))

    def test_names(self):
        """
        verifies the names of classes
        """
        for classname in CLASS_NAMES:
#            print classname
            self.assertTrue(classname in self.abilities_list)
#            for ability_name in ABILITY_NAMES:
#                print classname, ability_name
#                self.assertTrue(ability_name in \
#                                self.abilities_list[classname])


class TestSavingThrows(unittest.TestCase):
    """
    unittest TestCase to insure that the constant BASE_SAVING_THROWS is valid
    and properly filled.
        Each class must be listed once, and no others.
        Each class must have each of the 5 saving throws listed.
        Each saving throw value must be an integer that is between 1 and 19.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre_constants import BASE_SAVING_THROWS
        self.saving_throws = BASE_SAVING_THROWS

    def test_data_type(self):
        """
        verifies that the saving throws structure is a dictionary
        """
        self.assertTrue(isinstance(self.saving_throws, dict))

    def test_class_names(self):
        """
        verifies that each class is listed in the saving throws
        """
        for class_name in CLASS_NAMES:
            self.assertTrue(class_name in self.saving_throws)
        self.assertTrue(len(self.saving_throws) == len(CLASS_NAMES))

    def test_saving_throw_types(self):
        """
        verifies that each class has the proper saving throws
        """
        for class_name in self.saving_throws:
            self.assertTrue(isinstance(class_name, str))
            for saving_throw in SAVING_THROW_TYPES:
                self.assertTrue(isinstance(saving_throw, str))
                self.assertTrue(saving_throw in \
                                               self.saving_throws[class_name])
                self.assertTrue(len(self.saving_throws[class_name]) == \
                                len(SAVING_THROW_TYPES))
                self.assertTrue(isinstance( \
                           self.saving_throws[class_name][saving_throw], int))
                self.assertTrue( \
                            self.saving_throws[class_name][saving_throw] > 0)
                self.assertTrue( \
                            self.saving_throws[class_name][saving_throw] < 20)


class TestXP(unittest.TestCase):
    """
    unittest TestCase to insure that the constant XP_LEVELS is valid
    and properly filled.
        Each class must be listed once, and no others.
        Each XP level must be at least 20% higher than the previous one.
        Each saving throw value must be an integer that is between 1 and 19.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre_constants import XP_LEVELS
        self.xps = XP_LEVELS

    def test_monotonic(self):
        """
        test to insure that each XP level is greater than the previous one
        """
        for classname in self.xps:
            xplevels = self.xps[classname]
            maxlevel = max(xplevels)
            lastxp = 0
            for level in range(2, maxlevel):
                self.assertTrue(xplevels[level] > lastxp)
                lastxp = xplevels[level]

    def test_increasing_significantly(self):
        """
        test to insure that each level is at least 20% greater than the
        prevous one
        """
        for classname in self.xps:
            xplevels = self.xps[classname]
            maxlevel = max(xplevels)
            lastxp = 0
            for level in range(2, maxlevel):
                self.assertTrue(xplevels[level] >= 1.2 * lastxp)
                lastxp = xplevels[level]


class TestDie(unittest.TestCase):
    """
    unittest TestCase to test the Die class
        Tests using a 5d20 set of dice
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        num_die = 5
        die_sides = 20
        self.die = Die(str(num_die) + 'd' + str(die_sides))
        self.num_sides = die_sides
        self.num_dice = num_die

    def test_highest(self):
        """
        verifies that the Highest function returns the proper max value of
        rolling a Die object
        """
        max_possible = self.num_sides * self.num_dice
        self.assertEqual(self.die.highest(), max_possible)

    def test_lowest(self):
        """
        verifies that the Lowest function returns the proper min value of
        rolling a Die object
        """
        self.assertEqual(self.die.lowest(), self.num_dice)

    def test_rolls(self):
        """
        verifies the roll() function by the following criteria
            Each roll value is between the highest and lowest possible values
            The average of many rolls is close to the mean possibility
                within 0.1% (a flat distribution, not Gaussian)
            This function calls the roll function 5 x 20 x 100 = 10,000 times
        """
        # determine the number of rolls to test
        num_rolls = self.num_sides * self.num_dice * 100
        # test averages of rolls
        total = 0
        for _ in range(num_rolls):
            roll_value = self.die.roll()
            self.assertTrue(roll_value >= self.die.lowest())
            self.assertTrue(roll_value <= self.die.highest())
            total += roll_value
        meas_average = total / (num_rolls * 1.0)
        theo_average = (self.die.lowest() + self.die.highest()) / 2.0
        diff_err = (theo_average - meas_average) / meas_average
        self.assertAlmostEqual(0, diff_err, places=1)


# class testCharacter(unittest.TestCase):
#
#    def setUp(self):
#        self.
#
#    def testHighest(self):


if __name__ == '__main__':
    import doctest
    unittest.main()
    doctest.testmod()
