#!/usr/bin/env/python
"""
@package wyckedsceptre
wyckedsceptre is a Python 2.6.5 module for simulating encounters in
Dungeons & Dragons First Edition (Red Book - 1E) and predicting outcomes
of combat encounters.
This module is its authors first venture into active, open-source software
development. The author, Tim, is also an amateur Python developer looking
to get notes on his coding structure and style from experienced Python
developers as well as general notes about open-source development
and documentation.
@todo Make public members private

project home: http://code.google.com/p/wyckedsceptre/
Last Changed Date: $LastChangedDate: 2010-10-26 22:37:34 -0500 (Tue, 26 Oct 2$
Revision         : $Rev: 50 $
"""
import wyckedsceptre_constants


def assert_positive_integer(val_to_test, val_name='Value'):
    """
    Verifies that a value is a positive integer or zero.
    @param val_to_test Value to test if it is a positive integer or zero
    @param val_name A string name to describe that variable, to be used
            in the exception text if raised
    @except ValueError If val_to_test is < 0 or not an integer
    """
    if not isinstance(val_to_test, int) or val_to_test < 0:
        raise ValueError(val_name + ' must be a positive integer or zero')


class Die(object):
    """
    Object for simulating a set of an m-numbered set of n-sided dice

    >>> print wyckedsceptre.Die('1d20')
    1d20

    """

    def __init__(self, die_str='1d20'):
        """
        Constructor for the dice object
        @param die_str string that takes the form "MdN" where M is the number
        of dice and N is the number of sides each die has
        @todo add regular expression support die_str
        """
        if not isinstance(die_str, str):
            raise TypeError('Die constructor must take a string')
        dstr = die_str.split('d')
        ## the number of dice in a Die object (integer)
        self.num_die = int(dstr[0])
        ## the number of sides on a Die object (integer)
        self.die_sides = int(dstr[1])
        self.__invariant__()

    def __eq__(self, other):
        """
        overloaded equality test for Die object
        """
        if isinstance(other, Die):
            return ((self.num_die == other.num_die) and \
                    (self.die_sides == other.die_sides))
        else:
            return False

    def __invariant__(self):
        """
        Tests to insure the Die object's state is valid.
        Throws a ValueError exception if any problems are detected.
        @exception ValueError if # of sides or dice is invalid
        """
        if not (self.die_sides in wyckedsceptre_constants.VALID_DIE_SIDES):
            raise ValueError('Number of die sides must be either ' + \
                    str(wyckedsceptre_constants.VALID_DIE_SIDES))
        assert_positive_integer(self.num_die, 'Number of Dice')
        if 0 == self.num_die:
            raise ValueError('Number of die cannot be zero')

    def __str__(self):
        """
        String conversion routine for the Die object
        @returns a string version of the die object in MdN format
        example: 1d20
        """
        return str(self.num_die) + 'd' + str(self.die_sides)

    def highest(self):
        """
        The highest possible roll of a Die object
        @returns the highest possible roll of a Die object as an integer
        """
        return self.num_die * self.die_sides

    def lowest(self):
        """
        The lowest possible roll of a Die object
        @returns the lowest possible roll of a Die object as an integer
        """
        return self.num_die

    def roll(self):
        """
        Produces the summed result of a random roll of the dice
        Each die is rolled invididually
        @returns the sum of all die rolls as an integer
        """
        import random
        total = 0
        for _ in range(self.num_die):
            total += random.randint(1, self.die_sides)
        return total


class Attack(object):
    """
    An object for simulating a single RPG attack.
    The attack is composed of a Die object plus a bonus
    """

    def __init__(self, description='', die_roll='1d6', attack_bonus=0):
        """
        Constructor for the Attack object
        @param description - string that describes the attack
        @param die_roll     - string used in the Die object constructor
        @param attack_bonus - integer used as a static bonus to the
                      attack damage
        """
        ## a description of an attack (string)
        self._desc = description
        ## an integer describing the attack bonus
        self._bonus = attack_bonus
        ## a Die object that represents the attack (ex: 1d6)
        self._damage_die = Die(die_roll)

    def __eq__(self, other):
        """
        overloaded equality operator for Attack object
        """
        if isinstance(other, Attack):
            return ((self._desc == other._desc)
                and (self._bonus == other._bonus)
                and (self._damage_die == other._damage_die))
        else:
            return False

    def highest(self):
        """
        The highest possible outcome of an attack roll
        @returns the highest possible damage of an attack as an integer
        """
        return self._damage_die.highest() + self._bonus

    def lowest(self):
        """
        The lowest possible outcome of an attack roll
        @return the lowest possible damage of an attack as an integer
        """
        return self._damage_die.lowest() + self._bonus

    def roll_damage(self):
        """
        Rolls for damage and includes bonus
        @return a random damage roll for the Attack object as an integer
        """
        return self._damage_die.roll() + self._bonus


def abil_to_mod(ability_score):
    """
    Converts an ability score to its modifier. Negative modifiers are possible
    @param ability_score
    @returns the corresponding modifier for an ability score as an integer
    example: a Strength of 17 has a +3 strength modifier
    """
    return (ability_score - 10) // 2


class Character(object):
    """
    An object that describes an individual character
    """

    def __init__(self, new_name, new_class):
        """
        Constructor for the Character object
        @param new_name a String for the name of the character
        @param new_class a string for the class of the character
        @todo add full set of agruments
        @todo fill in Doxygen descriptions for each member
        @returns An initialized Character object
        """
        ## The name of the character (string)
        self.name = new_name
        ## The class of the character (string)
        self.dnd_class = new_class
        ## The character's level (integer)
        self.level = 1
        ## Character's current Hit Points (integer)
        self.hit_points = 10
        ## Character's maximum possible Hit Points (integer)
        self.hit_points_max = 10
        ## Character's Armor Class (integer)
        self.armor_class = 8
        ## Character's Initiative Bonus (integer)
        self.initiative_bonus = 0
        ## Character's current initiative roll, including bonus (integer)
        self.initiative_total = 0
        # self.inventory = list(items())
        ## Character's money in Gold Pieces (integer)
        self.money = 100
        ## Character's Experience (integer)
        self.xp = 0
        ## Bool - is this character a Non-Player Character (NPC)
        self.is_npc = True
        ## This character's attack (Attack object)
        self.attack = Attack()
        ## Character's core ability scores (dict)
        self.abilities = {'STR': 1, 'INT': 1, 'WIS': 1, 'DEX': 1, 'CON': 1,
                        'CHA': 1}
        ## Character's saving throws for each type (dict)
        self.saving_throws = {'DEATHRAY-POISON': 1, 'MAGICWANDS': 1,
                            'PARALYSIS-TURNSTONE': 1, 'DRAGONBREATH': 1,
                            'RODS-STAVES-SPELLS': 1}
        ## Character's state (integer between -1 and 1)
        self.state = 0.0
        self.update_state()

    def __eq__(self, other):
        """
        overloaded equality tester for Character object
        """
        if isinstance(other, Character):
            return ((self.name == other.name)
                # and (self.student_number == other.student_number)
                and (self.name == other.name)
                and (self.dnd_class == other.dnd_class)
                and (self.level == other.level)
                and (self.hit_points == other.hit_points)
                and (self.hit_points_max == other.hit_points_max)
                and (self.armor_class == other.armor_class)
                and (self.initiative_bonus == other.initiative_bonus)
                and (self.initiative_total == other.initiative_total)
                and (self.money == other.money)
                and (self.xp == other.xp)
                and (self.is_npc == other.is_npc)
                and (self.attack == other.attack)
                and (self.abilities == other.abilities)
                and (self.saving_throws == other.saving_throws)
                and (self.state == other.state))
        else:
            return False

    def __invariant__(self):
        """
        Tests to insure the Character object's state is valid.
        Throws a ValueError exception if any problems are detected.
        @exception ValueError a property is not within a valid range
        @exception TypeError a property is not the right data type
        """
        assert_positive_integer(self.xp, 'Character - XP')
        assert_positive_integer(self.armor_class, 'Character - Armor class')
        assert_positive_integer(self.money, 'Character - Money')
        assert_positive_integer(self.initiative_bonus, 'Initiative bonus')
        assert_positive_integer(self.initiative_total, 'Initiative total')
        assert_positive_integer(self.hit_points_max, 'Char - max hit points')
        assert_positive_integer(self.level, 'Character - level')
        if not isinstance(self.name, str):
            raise TypeError('Character name must be a string')
        if not isinstance(self.dnd_class, str):
            raise TypeError('Character class must be a string')
        if not isinstance(self.state, float):
            raise TypeError('Character state must be a float')
        if self.state > 1.0:
            raise TypeError('Character state must be less than 1, overhealed')

    def __str__(self):
        """
        String conversion routine
        @returns the character's name and a few stats on them
        """
        res = self.name + ' the level ' + str(self.level) + ' ' + \
                self.dnd_class + ' HP: ' + str(self.hit_points) + '/' + \
                str(self.hit_points_max)
        return res

    def get_hp(self):
        """
        The current hit points of a Character
        @returns Current HP of a Character (integer)
        """
        self.__invariant__()
        return self.hit_points

    def take_damage(self, amount):
        """
        Reduces the hit_points of a character by "amount".  Amount must be
        greater than zero
        @param amount The amount of damage inflicted on the character
                      A positive integer
        @exception If amount of damage is less than 1
        @returns the state after damage as a float [-1, 1]
        """
        self.__invariant__()
        assert_positive_integer(amount, 'Damage')
        self.hit_points -= amount
        self.update_state()
        self.__invariant__()
        return self.state

    def level_at_xp(self):
        """
        Computes the level of a character at their current XP level
        @return the level that corresponds to a given XP level
        """
        from wyckedsceptre_constants import XP_LEVELS
        assert_positive_integer(self.xp, 'Experience')
        self.__invariant__()
        curr_level = 1
        over = False
        while not over:
            if XP_LEVELS[self.dnd_class][curr_level + 1] < self.xp:
                curr_level += 1
            else:
                over = True
        return curr_level

    def heal(self, hit_points_to_heal):
        """
        Increases a character's hit points
        @param hit_points_to_heal Number of hit points to restore to a
        Character heal. Will not heal a Character who is completely dead.
        Will only heal a Character up to it's maximum hit_points
        @return the Character's new hit_points
        @exception if hit_points_to_heal is 0 or less
        """
        assert_positive_integer(hit_points_to_heal, 'Heal points')
        self.__invariant__()
        if self.get_hp() > -10: # won't heal totally dead people
            self.hit_points = min(self.hit_points_max,
                                    self.get_hp() + hit_points_to_heal)
            self.update_state()
        return self.get_hp()

    def is_concious(self):
        """
        Computes if the character is currently concious (hit_points > 0)
        @returns True if the character's hit_points is greater than zero.
        """
        self.update_state()
        return self.state > 0

    def update_state(self):
        """
        Updates the .state attribute based on hit_points, and checks for
        leveling up
        @returns the .state attribute [-1, 1]
        @exception if Character's current hit_points is greater than its max
                    hit_points
        """
        self.__invariant__()
        # force true division
        self.state = self.get_hp() / float(self.hit_points_max)
        self._level_up_()
        self.__invariant__()
        return self.state

    def get_ac(self):
        """
        Get the Armor Class of a character
        @return the armor class for a character as an integer
        """
        assert_positive_integer(self.armor_class, 'Armor Class')
        return self.armor_class

    def _level_up_(self):
        """
        Sets the character's level according to their XP
        @return true if the level changed
        @return false if no level change occured
        """
        self.__invariant__()
        new_level = self.level_at_xp()
        if new_level != self.level:
            self.level = new_level
            print self.name + ' has leveled up to level # ' + str(self.level)
            return True
        else:
            return False

    def roll_initiative(self):
        """
        Rolls and returns initiative for the character
        @returns a 1d20 + initiative bonus roll for a character's initiative
        @todo set the total_initiative member
        """
        self.__invariant__()
        self.initiative_total = self.initiative_bonus + Die('1d20').roll()
        self.__invariant__()
        return self.initiative_total

    def get_mod(self, ability):
        """
        Gets a skill modifier for a particular skill
        @returns the ability modifier for a given ability.
        @param ability A 3 character string, in all caps, that
                describes which ability
        """
        return abil_to_mod(self.abilities[ability])

    def make_attack(self):
        """
        Makes an attack roll - hit and damage
        @return a list of integers [Hit, Damage]
        where Hit is a 1d20 attack roll and Damage is the damage roll done
        if the target is hit
        @todo add ability modifiers to attack and damage rolls
        """
        hit_die = Die('1d20')
        hit_roll = hit_die.roll() + self.get_mod('STR')
        return [hit_roll, self.attack.roll_damage()]


def roll_new_character(new_name, new_class):
    """
    Creates a new random character
    @param NewName A string that names the character
    @param NewClass A string that is the class of the character
    @returns a new Character object
    """
    new_char = Character(new_name, new_class)
    die_1d6 = Die('1d6')
    for abil in new_char.abilities.keys(): # roll abilities
        rolls = [0] * 4
        for j in range(len(rolls)):
            rolls[j] = die_1d6.roll()
        new_char.abilities[abil] = sum(rolls) - min(rolls)
    return new_char


class Encounter(object):
    """
    An object that describes a combat encounter between two parties
    """

    def __init__(self):
        """
        Default constructor for Encounter object
        sets all members to empty, no players or npc's
        @returns an initialized Encounter object
        """
        ## The list of Character objects that are Player controlled
        self.players = list()
        ## The list of Character objects that are NPC's
        self.npcs = list()
        ## The list of Character objects in order of initiative
        self.run_order = list()
        ## An iterator pointing to the current player's turn
        self.active_turn = iter(self.run_order)

    def run_encounter(self):
        """
        Start simulating an encounter after it has been properly initialized
        Will simulate the encounter until all members of one of the two
        parties are unconcious or dead
        @todo implement run_encounter
        """
        # roll initiatives
        # while both parties have concious people
        #  __run_round__
        # return results
        pass

    def __run_round__(self):
        """
        Private function that simulates an individual round in an encounter
        RunRound iterates through each character's turn according to
        the run order.
        Assumes: initiatives have already been rolled
                 the characters have been ordered accordingly
        @todo implement __run_round__
        """
        # iterate thru each character's turn
        #    check if round or continuing is necessary
        #    make an attack on another character
        #    apply damage to target
        # update each player's status
        pass

    def add_character(self, new_char, group):
        """
        Adds a single character to an Encounter object.
        @param NewChar A Character object to add to the encounter
        @param group A string that is either 'players' or 'npcs'
        @returns True if the character was added properly
        @returns False if the character couldn't be added or 'group' is not
        either 'players' or 'npcs'
        @todo add return values
        """
        if 'players' == group:
            self.players.insert(new_char)
        elif 'npcs' == group:
            self.npcs.insert(new_char)
#        else:
#            raise Encounter.GroupNotSpecified

    def players_any_concious(self):
        """
        Determines if any Player characters are currently concious
        @returns True if at least 1 member of the players group is concious
        @returns False if there are no members in the players group or if none
        are concious
        """
        for member in self.players:
            if member.is_concious():
                return True
        return False

    def npcs_any_concious(self):
        """
        Determines if any NPC characters are currently concious
        @returns True if at least 1 member of the npcs group is concious
        @returns False if there are no members in the npcs group or if none
        are concious
        """
        for member in self.npcs:
            if member.is_concious():
                return True
        return False

    def roll_initiatives(self):
        """
        Rolls initiatives for each character in the encounter and sorts them
        according to initiative.
        @todo need pointers to character objs
        """
        init_rolls = []
        for character in self.run_order:
            init_rolls.append(character.roll_initiative())
        self.run_order.sort(key=lambda x: x.initiative_total, reverse=True)


def load_file(filename):
    """
    Loads a single object from a save file in pickle format
    @param filename the string path to a file in pickle format
    @returns The object if it exists in the file
    @returns False if the object/file could not be read correctly
    """
    try:
        import pickle
        file_handler = open(filename, 'r')
    except IOError:
        pass
    finally:
        if file_handler:
            res = pickle.load(file_handler)
            file_handler.close()
            return res
        else:
            return False


def save_file(obj, filename):
    """
    Saves an object to a pickle file
    @params obj An object or variable to save out in pickle format
    @params filename The path to a file to save to (overwrite)
    @returns True if the file was written correctly
    @returns False if the file could not be written to properly
    """
    # ex: "bob.char.p" ; "forrest_fight.enc.p"
    # http://wiki.python.org/moin/UsingPickle
    try:
        import pickle
        res = False
        file_handler = open(filename, 'w')
        pickle.dump(obj, file_handler)
        res = True
    except IOError:
        pass
    finally:
        file_handler.close()
        return res
