#!/usr/bin/env/python
# wyckedsceptre_test.py
"""
@package wyckedsceptre_test
Module based on Python's unittest to test each of the classes
created in the wyckedsceptre module, as well as the constants defined
in wyckedsceptre_constants.py

project home: http://code.google.com/p/wyckedsceptre/
Last Changed Date: $LastChangedDate: 2010-10-26 22:37:34 -0500 (Tue, 26 Oct 2$
Revision         : $Rev: 50 $
"""
import unittest
import sys
import os

## Add the wyckedsceptre main directory to current Python path
## This enables the "from wyckedsceptre.wyckedsceptre_constants" syntax
PATH_TO_WS_MAIN = os.path.realpath(__file__)
(PATH_TO_WS_MAIN, _) = os.path.split(PATH_TO_WS_MAIN)
PATH_TO_WS_MAIN = os.path.join(PATH_TO_WS_MAIN, "..")
sys.path.append(PATH_TO_WS_MAIN)
from wyckedsceptre.wyckedsceptre import Die, Attack, load_file
from wyckedsceptre.wyckedsceptre import roll_new_character, save_file


## Constant lists to be used for testing
CLASS_NAMES = ['cleric', 'fighter', 'thief', 'dwarf', 'elf', 'halfling',
                'magicuser']

SAVING_THROW_TYPES = ['DEATHRAY-POISON', 'MAGICWANDS', 'PARALYSIS-TURNSTONE',
                      'DRAGONBREATH', 'RODS-STAVES-SPELLS']

ABILITY_NAMES = ['STR', 'INT', 'WIS', 'DEX', 'CON', 'CHA']


class TestAbilityScoreOrder(unittest.TestCase):
    """
    unittest TestCase to insure that the constant ABIL_SCORE_ORDER is valid
    and properly filled.  Each class must be listed once, and no others.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre.wyckedsceptre_constants import ABIL_SCORE_ORDER
        self.abilities_list = ABIL_SCORE_ORDER

    def test_data_type(self):
        """
        verifies that each class is a dictionary in the ability scores
        """
        self.assertTrue(isinstance(self.abilities_list, dict))

    def test_names(self):
        """
        verifies the names of classes
        """
        for classname in CLASS_NAMES:
#            print classname
            self.assertTrue(classname in self.abilities_list)
#            for ability_name in ABILITY_NAMES:
#                print classname, ability_name
#                self.assertTrue(ability_name in \
#                                self.abilities_list[classname])


class TestSavingThrows(unittest.TestCase):
    """
    unittest TestCase to insure that the constant BASE_SAVING_THROWS is valid
    and properly filled.
        Each class must be listed once, and no others.
        Each class must have each of the 5 saving throws listed.
        Each saving throw value must be an integer that is between 1 and 19.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre.wyckedsceptre_constants import BASE_SAVING_THROWS
        self.saving_throws = BASE_SAVING_THROWS

    def test_data_type(self):
        """
        verifies that the saving throws structure is a dictionary
        """
        self.assertTrue(isinstance(self.saving_throws, dict))

    def test_class_names(self):
        """
        verifies that each class is listed in the saving throws
        """
        for class_name in CLASS_NAMES:
            self.assertTrue(class_name in self.saving_throws)
        self.assertTrue(len(self.saving_throws) == len(CLASS_NAMES))

    def test_saving_throw_types(self):
        """
        verifies that each class has the proper saving throws
        """
        for class_name in self.saving_throws:
            self.assertTrue(isinstance(class_name, str))
            for saving_throw in SAVING_THROW_TYPES:
                self.assertTrue(isinstance(saving_throw, str))
                self.assertTrue(saving_throw in \
                                               self.saving_throws[class_name])
                self.assertTrue(len(self.saving_throws[class_name]) == \
                                len(SAVING_THROW_TYPES))
                self.assertTrue(isinstance( \
                           self.saving_throws[class_name][saving_throw], int))
                self.assertTrue( \
                            self.saving_throws[class_name][saving_throw] > 0)
                self.assertTrue( \
                            self.saving_throws[class_name][saving_throw] < 20)


class TestXP(unittest.TestCase):
    """
    unittest TestCase to insure that the constant XP_LEVELS is valid
    and properly filled.
        Each class must be listed once, and no others.
        Each XP level must be at least 20% higher than the previous one.
        Each saving throw value must be an integer that is between 1 and 19.
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        from wyckedsceptre.wyckedsceptre_constants import XP_LEVELS
        self.xps = XP_LEVELS

    def test_monotonic(self):
        """
        test to insure that each XP level is greater than the previous one
        """
        for classname in self.xps:
            xplevels = self.xps[classname]
            maxlevel = max(xplevels)
            lastxp = 0
            for level in range(2, maxlevel):
                self.assertTrue(xplevels[level] > lastxp)
                lastxp = xplevels[level]

    def test_increasing_significantly(self):
        """
        test to insure that each level is at least 20% greater than the
        prevous one
        """
        for classname in self.xps:
            xplevels = self.xps[classname]
            maxlevel = max(xplevels)
            lastxp = 0
            for level in range(2, maxlevel):
                self.assertTrue(xplevels[level] >= 1.2 * lastxp)
                lastxp = xplevels[level]


class TestDie(unittest.TestCase):
    """
    unittest TestCase to test the Die class
        Tests using a 5d20 set of dice
    """

    def setUp(self):
        """
        setUp function saught by unittest
        sets up for testing
        """
        num_die = 5
        die_sides = 20
        self.die = Die(str(num_die) + 'd' + str(die_sides))
        self.num_sides = die_sides
        self.num_dice = num_die

    def test_equality(self):
        """
        tests that equality (__eq__ or ==) works
        """
        self.assertEqual(self.die, self.die) ## idenity
        self.assertEqual(self.die, Die(str(self.num_dice) + 'd' + \
                        str(self.num_sides)))
        self.assertEqual(Die(str(self.num_dice) + 'd' + str(self.num_sides)),
                         self.die)

    def test_highest(self):
        """
        verifies that the Highest function returns the proper max value of
        rolling a Die object
        """
        max_possible = self.num_sides * self.num_dice
        self.assertEqual(self.die.highest(), max_possible)

    def test_lowest(self):
        """
        verifies that the Lowest function returns the proper min value of
        rolling a Die object
        """
        self.assertEqual(self.die.lowest(), self.num_dice)

    def test_rolls(self):
        """
        verifies the roll() function by the following criteria
            Each roll value is between the highest and lowest possible values
            The average of many rolls is close to the mean possibility
                within 0.1% (a flat distribution, not Gaussian)
            This function calls the roll function 5 x 20 x 100 = 10,000 times
        """
        # determine the number of rolls to test
        num_rolls = self.num_sides * self.num_dice * 100
        # test averages of rolls
        total = 0
        for _ in range(num_rolls):
            roll_value = self.die.roll()
            self.assertTrue(roll_value >= self.die.lowest())
            self.assertTrue(roll_value <= self.die.highest())
            total += roll_value
        # force true division
        meas_average = total / float(num_rolls)
        theo_average = (self.die.lowest() + self.die.highest()) / 2.0
        diff_err = (theo_average - meas_average) / meas_average
        self.assertAlmostEqual(0, diff_err, places=1)


class TestLoadSave(unittest.TestCase):
    """
    tests load_data and save_data by creating a character, saving it,
    then reloading it and making sure it is the same
    """

    def setUp(self):
        """
        setup the test for load and save
        """
        mace = Attack('mace', '1d6', 1)
        bob = roll_new_character('bob', 'magicuser')
        bob.attack = mace
        self.test_character = bob
        self.test_filename = 'test.pickle'
        if os.path.isfile(self.test_filename):
            os.remove(self.test_filename)

    def test_load_save(self):
        """
        saves a character out, then reloads it and compares them to make sure
        they are the same
        """
        self.assertTrue(save_file(self.test_character,
                                    self.test_filename))
        self.assertTrue(os.path.isfile(self.test_filename))
        loaded_character = load_file(self.test_filename)
        self.assertEqual(loaded_character, self.test_character)
        self.assertEqual(self.test_character, loaded_character)
        if os.path.isfile(self.test_filename): ## clean up
            os.remove(self.test_filename)

# class testCharacter(unittest.TestCase):
#
#    def setUp(self):
#        self.
#
#    def testHighest(self):


if __name__ == '__main__':
    import doctest
    unittest.main()
    doctest.testmod()
