# -*- coding: utf-8 -*-
# WinspecUtils.py module, copied and extended by KJR from
# piUtils.py module "Princeton Instruments Utilities"
# (renamed because I removed the function converting to "fits" files
#   so that I could remove the pyfits dependency)
#
# Process SPE data generated by Princeton Instruments cameras
# From original author (James Battat):
#    If you improve/extend the code, I'd love to hear about it.
#    Please contact me at:
#       James Battat
#       jbattat@post.harvard.edu
#    2009 April
#
# Revised/Extended (with fits conversion removed)
# May 2010:
# Extended to handle spectroscopy data (x-calibration) and 
# Spectrum class added for convenience.
# Kasey Russell (krussell at post.harvard.edu)
#
# Some yoffset features added by Shanying Cui
# Kasey Russell (krussell@post.harvard.edu)
#

# Changelog from ximeg
# 
# + comments are read and saved
# + calibration turned off
# 
#
#
#

from __future__ import division

# the end of the version is the date: year-mo-day
__version__ = "0.0.1 r2012-08-18"

import sys
import os.path
import struct # deal with binary data
import numpy
import pylab
import textwrap

class Spectrum():
    """
    This class loads in a Winspec file (*.SPE*) and stores the wavelength
    and luminescence as self.wavelen (in nm) and self.lum (in counts or
    counts per second), respectively.
    Various methods allow for plotting the data [*self.plot()*],
    normalizing it [*self.normalize()*], etc.
    """
    def __init__( self, fname, cts_per_sec=False, verbose=False):
        if not (fname.endswith('.SPE') or fname.endswith('.TXT')
             or fname.endswith('.spe') or fname.endswith('.txt')):
            raise ValueError( 'Can only create a Spectrum class object from an SPE or txt\
                                                                  file; fname=%s' % fname )
            
        self.fname          = fname
        # Force load counts so self.raw_lum is counts, then correct counts/sec later
        self.cts_per_sec    = False
        self.ax             = None
        self.fit_params     = None
        self.laser          = None
        self._slope_npoints = None
        self._slope_remove  = False
        self._counts_per_second = False

        if (fname.endswith('.SPE') or fname.endswith('.spe')):
            self.read_spe(verbose=verbose)
        if (fname.endswith('.TXT') or fname.endswith('.txt')):
            self.read_txt()
        self.raw_lum = self.lum.copy()
        if cts_per_sec:
            self.counts_per_second()

    def background_correct( self, fname  ):
        """
        for an already-loaded spectrum, remove the background spectrum
        contained in the file fname.
        This won't change the flag self.background_corrected, which is meant
        as an indicator of whether the file was corrected during acquisition or not.
        """
        self.fname_bg = fname
        self.wavelen_bg, self.lum_bg = get_spectrum( fname, self.cts_per_sec )
        self.lum -= self.lum_bg
        

    def chop( self, start=None, stop=None ):
        """
        restrict the spectrum to only this wavelength range
        (start and stop are in nm)
        """
        if start is None:
            start = self.wavelen[0]
        
        if stop is None:
            stop = self.wavelen[-1]
        
        if start < self.wavelen[0] or stop > self.wavelen[-1]:
            raise ValueError("wavelength range is not contained within\
                                the spectrum.")
        
        self.lum = self.lum[ pylab.find(self.wavelen > start) ]
        self.wavelen = self.wavelen[ find(self.wavelen > start) ]
        
        self.lum = self.lum[ pylab.find(self.wavelen < stop) ]
        self.wavelen = self.wavelen[ find(self.wavelen < stop) ]


    def counts_per_second( self ):
        """ If you didn't load the data using 'cts_per_sec=True', this
        method will divide the luminescence intensity by the exposure time
        to correct it. If you did load it with 'cts_per_sec=True', then
        this method does nothing. """
        if not self._counts_per_second:
            self.lum /= self.exposure
            self._counts_per_second = True
        

    def fit_gaussians( self, plotfit=True, halfwidth=5.0, width=None,
                       center=None, height=None, yoffset=0.0, printparams=True ):
        """
        By default (i.e. unless 'center' is specified), this fits Gaussians
        using the function *easyfitgaussians* from the module
        *kasey_fitspectra*. This will then ask you to click on the peaks you
        would like to fit.
        
        The parameter 'halfwidth' specifies the initial guess at the peak
        halfwidth (in nm). 
        
        Alternatively, you can pass the keyword argument 'width'
        (equal to twice the halfwidth). If both are specified, the
        value for 'width' will be used.

        The fit parameters are stored in self.fitparams as a
        list of dictionaries, one for each peak, with the following
        keys:
            
            'yoffset', 'ymax', 'halfwidth', 'x0'

        where yoffset and ymax are the vertical/lum. offset and
        the lum. peak height, respectively,
        and x0 is the peak wavelength
        (by halfwidth, I mean standard deviation.)

        By default (plotfit=True), it plots the result as a black line
        of width 2. To override this, pass plotfit=False.

        If 'center' is specified, then the fit will not be interactive, and
        the data will be fit with a single lorentzian centered at 'center' and
        of amplitude 'height' (which, if not specified, will be taken as the
        value of the data at the point 'center').

        By default (printparams=True), the fit parameters are printed to screen.
        """

        if center is None:
            withmouse = True
        else:
            withmouse = False
            if center < self.wavelen.min():
                center = self.wavelen.min()
            if center > self.wavelen.max():
                center = self.wavelen.max()
            
        if width is not None:
            halfwidth = width/2
            
        if withmouse==True:
            result = kcfit.easyfitgaussians( self.wavelen, self.lum, 
                                               halfwidth )
            self.bestfit, self.fit_params = result
            
        else:
            #For a single Gaussian, params can be a dict containing
            #the keys yoffset, ymax, x0, and halfwidth.
            if height is None:
                dw = self.wavelen[1] - self.wavelen[0]
                height = self.lum[ pylab.where( abs(self.wavelen-center)<dw/2 ) ]
            
            initparams = dict( yoffset=yoffset,
                               ymax=height,
                               halfwidth=halfwidth,
                               x0=center )

            result = kcfit.fitgaussians( self.wavelen, self.lum, initparams )
            self.bestfit, pbest, std_err = result
            
            kcfit.absparams(pbest)
            self.fit_params = kcfit.params_to_dicts( kcfit.splitparams(pbest), 
                                                    kcfit.splitparams(std_err) )

        self.fit_wavelen = numpy.linspace(self.wavelen.min(), self.wavelen.max(),
                                          len(self.wavelen))

        if plotfit==True:
            self.plot_fit( linewidth=2.0, color='black' )

        if printparams==True:
            self.print_fit_parameters()
    
    
    def fit_lorentzians( self, plotfit=True, halfwidth=5.0, width=None,
                       center=None, height=None, yoffset=0.0, printparams=True,
                       fitrange=(0,-1), number_bestfit_points=None ):
        """
        Fit one or more Lorentzian functions to the data. If you use the *center* parameter,
        the fit will not be interactive and will attempt to fit based on the info passed in.
        If you do not use the *center* parameter, then the function *easyfitlorentzians* from
        the module *kasey_fitspectra* will be used, and it will ask you to click on peaks that
        you would like to fit. That case does not allow you to restrict the fitting range,
        though, so the entire spectrum must be fit, which is often not what you want.
        
        *plotfit*
            Boolean value (default=True) plots the resulting fit to the current axes.

        *halfwidth*
            Guess at peak halfwidth in nm (default=5.0). For multiple peaks,
            a single value can be used for all peaks or a list of values can
            be specified, one for each peak.
            Alternatively, you can pass the keyword argument
            *width* (equal to twice the halfwidth). If both are specified, the
            value for 'width' will be used.
                    
        *width*
            Guess at peak width in nm. If both *width* and *halfwidth* are specified,
            *width* will be used.

        *center*
            Guess at center point in nm for the peak. For multiple peaks, a
            list of values should be supplied.
            
        *height*
            Guess at height of peak. If not supplied, will be taken as the
            value of the data at the point *center*.

        *yoffset*
            Guess at y-axis offset of the base of the Lorentzian. Default is 0.
            
        *printparams*
            Boolean value (default=True) specifying whether the resulting fit parameters should
            be printed to the terminal.

        *fitrange*
            Two-element tuple (*start, stop*) containing the array indices specifying
            the beginning and end of the fit range. Default is (0,-1), the full range.
        
        *number_bestfit_points*
            The number of points to use in the best fit line. Default is to use the same
            number as the data. You might want more if you have a really sharp peak and
            want to have a smooth Lorentzian curve over it, for example.

        The fit parameters are stored in self.fitparams as a
        list of dictionaries, one for each peak, with the following
        keys:
            *yoffset*, *ymax*, *halfwidth*, *x0*, *Q*
        where *yoffset* and *ymax* are the vertical/lum. offset and
        the lum. peak height, respectively,
        and *x0* is the peak wavelength

        """

        if center is None:
            withmouse = True
            if width is not None: halfwidth = width/2
        else:
            withmouse = False
            if len(numpy.shape(center)) == 0: center = [center]
            if len(numpy.shape(halfwidth)) == 0: halfwidth = [halfwidth]*len(center)
            if len(numpy.shape(yoffset)) == 0: yoffset = [yoffset]*len(center)
            if width is not None:
                if len(numpy.shape(width)) == 0:
                    halfwidth = [width/2]*len(center)
                else:
                    halfwidth = [item/2 for item in width]
                
            for i,item in enumerate(center):
                if item < self.wavelen.min(): center[i] = self.wavelen.min()
                if item > self.wavelen.max(): center[i] = self.wavelen.max()
            
            
        if withmouse==True:
            result = kcfit.easyfitlorentzians( self.wavelen, self.lum, 
                                               halfwidth )
            self.bestfit, self.fit_params = result
            
        else:
            """
            The initial paramters is a list of dicts, with each dict
            containing the parameters for a single lorentzian with
            the keys yoffset, ymax, x0, and halfwidth.
            """ 

            if height is None:
                dw = self.wavelen[1] - self.wavelen[0]
                height = [self.lum[ pylab.where( abs(self.wavelen-x0)<dw/2 ) ] for x0 in center]
            elif len(numpy.shape(height))==0:
                height = [height]
            
            initparams = []
            for y0, ymax, dx, x0 in zip(yoffset, height, halfwidth, center):
                initparams.append(dict( yoffset=y0,
                                   ymax=ymax,
                                   halfwidth=dx,
                                   x0=x0 ))

            """ 'values' is the array of y-axis values """
            values = self.lum[fitrange[0]:fitrange[1]].copy()


            """ optionally remove a linear bg before fitting lorentzians """
            if self._slope_remove:
                npoints = self._slope_npoints
                subset_x = self.wavelen[fitrange[0]:fitrange[1]]
                subset_y = self.lum[fitrange[0]:fitrange[1]]
                if npoints is None:
                    linearfit = numpy.polyfit( 
                        self.wavelen[fitrange[0]:fitrange[1]], 
                        self.lum[fitrange[0]:fitrange[1]], 
                        deg=1 )
                else:
                    x = numpy.zeros( 2*npoints )
                    x[:npoints] = subset_x[:npoints]
                    x[npoints:] = subset_x[-npoints:]
                    
                    y = numpy.zeros( 2*npoints )
                    y[:npoints] = subset_y[:npoints]
                    y[npoints:] = subset_y[-npoints:]
                    
                    linearfit = numpy.polyfit( x, y, 1 )

                linear_bg = numpy.polyval( linearfit, self.wavelen[fitrange[0]:fitrange[1]] )
                values -= linear_bg


            """ Do the actual Lorentzian fit. """
            bestfit, self.pbest, std_err = kcfit.fitlorentzians(
                self.wavelen[fitrange[0]:fitrange[1]],
                values, 
                initparams )
            
            """ Optionally increase the resolution of the bestfit points """
            if number_bestfit_points is None:
                self.bestfit = bestfit
                self.fit_wavelen = self.wavelen
            else:
                self.fit_wavelen = numpy.linspace(
                    self.wavelen[fitrange[0]],
                    self.wavelen[fitrange[1]],
                    number_bestfit_points)
                self.bestfit = kcfit.lorentzians(self.fit_wavelen, *self.pbest)


            """ Make all parameters > 0 for easy ordering (those squared within the function
            can come back as +/-). I abs() all of them out of lasiness, but this would be bad if we
            had a negative yoffset or a dip rather than a peak. You don't really need to do this, 
            though, so if it causes problems just comment it out."""
            kcfit.absparams(self.pbest)
            self.fit_params = kcfit.params_to_dicts( kcfit.splitparams(self.pbest), 
                                                    kcfit.splitparams(std_err) )
            
            for peak in self.fit_params:
                peak['Q'] = peak['x0']/2/peak['halfwidth']
                # error in Q is returned as a min/max tuple of the standard error
                peak['Q_err'] = numpy.asarray( (peak['x0']-peak['x0_err'])/2/(peak['halfwidth']+peak['halfwidth_err']),
                                         (peak['x0']+peak['x0_err'])/2/(peak['halfwidth']-peak['halfwidth_err']) )
                peak['Q_err'] = numpy.abs( peak['Q'] - peak['Q_err'] )
        
                if self._slope_remove:
                    peak['slope_removed'] = True
                    peak['slope_npoints'] = self._slope_npoints
                    peak['slope_fit'] = linearfit
                

        if plotfit==True:
            self.plot_fit( linewidth=2.0, color='black', fitrange=fitrange )

        if printparams==True:
            self.print_fit_parameters()
    
    def get_fitrange_indices( self, fitrange_nm ):
        """
        Convert a wavelength range (*start*, *stop*) into a range of array indices.
        This algorithm doesn't check that your input is valid and doesn't try to
        figure out which point you're closest to, only the largest one you are larger than.
        """
        start, stop = fitrange_nm
        if start >= self.wavelen[-1]:
            raise ValueError ("Start value too high.")
        start_index = pylab.where(self.wavelen > start)[0][0]

        if stop < self.wavelen[-1]:
            stop_index = pylab.where(self.wavelen > stop)[0][0]
        else:
            stop_index = -1

        return start_index, stop_index

    def normalize( self, wavelen=None, value=None ):
        """ Multiplies the spectrum by 'value'.
            
            If value is None, value is set to 1/lum.max() to
            normalize the intensity to unity.
            
            If wavelen=None (default), this just divides the
            luminescence intensity by its peak value.
            
            If wavelen is specified, then the intensity is
            normalized to the intensity at that wavelength
            rather than at the peak.

            To specify an endpoint of the spectrum, simply
            set wavelen outside the wavelength range:
            if wavelen is less than self.wavelen.min(), then
            self.wavelen.min() is used (and ditto for max value).
            
            The value multiplied to normalize the spectrum will
            be stored in self.scale_factor, e.g. 1/lum.max() by default.
            
        """
        if wavelen is None and value is None:
            self.scale_factor = 1.0/self.lum.max()
            self.lum /= self.lum.max()
        elif wavelen is not None:
            if wavelen < self.wavelen.min():
                self.lum /= self.lum[0]
                self.scale_factor = 1.0/self.lum[0]
            elif wavelen > self.wavelen.max():
                self.lum /= self.lum[-1]
                self.scale_factor = 1.0/self.lum[-1]
            else:
                lum_i = find( self.wavelen < wavelen )[-1]
                self.lum /= self.lum[ lum_i ]
                self.scale_factor = 1.0/self.lum[ lum_i ]
        elif value is not None:
            self.lum *= value
            self.scale_factor = value
        

    def on_close_axes( self, event ):
        """ Here we reset the self.axes flag to None so that if
        you run the *plot* method again you will get a new window
        rather than plot into a closed plot window... """
        #event.canvas.figure.axes[0].has_been_closed = True
        self.ax = None


    def plot( self, *args, **kwargs ):
        """
        Plots the spectrum.
        Any extra arguments passed to this method get passed
        on to plot, making it possible to do stuff like
        self.plot( '--b', linewidth=3 )
        You can add a vertical offset to the data by setting
        yoffset = xxx
        or a horizontal offset with
        xoffset = xxx
        (e.g. to enable making a waterfall plot of many traces)
        
        if you pass fill=True, then the function 'fill_between' will
        be used instead of 'plot', and the value of 'yoffset' (if set)
        will be used as the 'y2' value for 'fill_between'
        
        if you pass as_raman=True then it will be plotted as a Raman spectrum
        (i.e. in relative wavenumbers) BUT make sure you first set the laser
        wavelength (in nm) as in:
        
        self.laser = 532
        self.plot( as_raman=True )
        """
        if 'yoffset' in kwargs.keys():
            voffset = kwargs['yoffset']
            del kwargs['yoffset']
        else:
            voffset = 0.0
            
        if 'xoffset' in kwargs.keys():
            hoffset = kwargs['xoffset']
            del kwargs['xoffset']
        else:
            hoffset = 0.0
            
        if 'fill' in kwargs.keys():
            fill = kwargs['fill']
            del kwargs['fill']
        else:
            fill = False
        
        if 'as_raman' in kwargs.keys():
            as_raman=kwargs['as_raman']
            del kwargs['as_raman']
            if as_raman==True and self.laser is None:
                raise ValueError('Laser wavelength not set for plotting relative wavenumbers.')
        else:
            as_raman=False
        
        if 'semilogy' in kwargs.keys():
            semilogy = kwargs['semilogy']
            del kwargs['semilogy']
        else:
            semilogy = False

        if 'connect_on_close' in kwargs.keys():
            connect_on_close = kwargs['connect_on_close']
            del kwargs['connect_on_close']
        else:
            connect_on_close = True

        if as_raman == True:
            x_axis = hoffset + (1.0/self.laser - 1.0/self.wavelen)*1.0e7
        else:
            x_axis = self.wavelen + hoffset
                
        if self.ax is None:
            self.ax = pylab.gca()
            
        if fill:
            line = self.ax.fill_between( x_axis, self.lum + voffset,
                         y2 = voffset, **kwargs )
        else:
            line, = self.ax.plot( x_axis, self.lum + voffset,
                         *args, **kwargs )

        if semilogy:
            self.ax.set_yscale('log')

        if as_raman==True:
            self.ax.set_xlabel('Raman shift (cm$^{-1}$)')
        else:
            self.ax.set_xlabel('Pixel number')  # ximeg
            
        if (self.cts_per_sec or self._counts_per_second) and not self.flatfield_corrected:
            # I don't yet understand what the intensity units are
            # when using flatfield correction...
            self.ax.set_ylabel('Intensity (counts/s)')
        else:
            self.ax.set_ylabel('Intensity (counts)')
            self.ax.yaxis.major.formatter.set_powerlimits((-3, 4))

        if connect_on_close:
            pylab.gcf().canvas.mpl_connect('close_event', self.on_close_axes)
        self.ax.grid(True)
        self.ax.set_xlim(self.wavelen.min(), self.wavelen.max())

        return line

    def plot_fit( self, *args, **kwargs ):
        """
        If you fitted one (or more) Lorentzians or Gaussians to your spectrum,
        this method will plot them. As with the plot method, any
        extra args passed to this method get passed directly to
        pylab's plot command.
        """
        try:
            self.bestfit
        except (AttributeError, NameError):
            print "no fit found."
            return
        
        if 'yoffset' in kwargs.keys():
            voffset = kwargs['yoffset']
            del kwargs['yoffset']
        else:
            voffset = 0.0
            
        if 'xoffset' in kwargs.keys():
            hoffset = kwargs['xoffset']
            del kwargs['xoffset']
        else:
            hoffset = 0.0
            
        if 'fitrange' in kwargs.keys():
            fitrange = kwargs['fitrange']
            del kwargs['fitrange']
        else:
            fitrange = (0,-1)

        if self._slope_remove:
            """ restore the linear background """
            linear_bg = numpy.polyval( self.fit_params[0]['slope_fit'], self.fit_wavelen )
            voffset += linear_bg
            
        line, = pylab.plot( self.fit_wavelen+hoffset, self.bestfit+voffset, *args, **kwargs )
        return line
        

    def print_fit_parameters( self ):
        """ print the fit parameters to the screen in a nice format
        """
        if self.fit_params is None:
            print "There are no fit parameters to display."
            return
        
        peak_number = 1
        for peak in self.fit_params:
            print "Peak number %d:" % (peak_number)
            print "  Wavelength: %.2f +- %.2f" % (peak['x0'], peak['x0_err'])
            print "      Height: %.2f +- %.2f" % (peak['ymax'], peak['ymax_err'])
            if 'Q' in peak.keys():
                print "           Q: %.2f +- %.2f" % (peak['Q'], peak['Q_err'])
            print "      Offset: %.2f +- %.2f" % (peak['yoffset'], peak['yoffset_err'])
            print ""
            
            peak_number += 1

    # Дурость-дуростью
    def read_spe( self, cts_per_sec=False, verbose=False):
        """ 
        Read in an SPE file.
        Saves lum. and wavelength in self.lum and self.wavelen.
        Saves SPE file header info in self.header        
        If you pass it the additional argument
        cts_per_sec=True
        then it will return the spectrum in counts per second rather than
        just counts (the default).
        """
         # calls the function defined outside this class
        spe_struct, self.fileinfo = read_spe( self.fname, verbose=verbose )
        self.lum = numpy.transpose( spe_struct['data'][0] )

        # the luminescence is stored as a 2D matrix (capable of holding each pixel in the CCD array)
        # even though it's only 1D of data (assuming you're in 'spectrum' mode and 
        # thus that the columns of the CCD were integrated)
        # Here we flatten it to a 1D array:        
        self.lum = self.lum.flatten()
            
        self.comments = spe_struct['COMMENTS']
        self.exposure = spe_struct['EXPOSURE']
        self.accumulations = spe_struct['ACCUMULATIONS']
        if cts_per_sec:
            """Normalize to counts per second."""
            self.lum /= ( self.exposure * self.accumulations )

        xcal = spe_struct['XCALIB']

        self.flatfield_corrected = spe_struct['FLATFIELD']
        self.background_corrected = spe_struct['BACKGROUND']
        self.grating_grooves = xcal['SpecGrooves']
        self.spe_struct = spe_struct


# I commented out the calibration routine # ximeg
        self.wavelen = numpy.arange(0, len(self.lum))
#        if xcal['SpecGlueFlag']:
#            # spectrum was 'glued' from spectra at different grating angles
#            step=xcal['SpecGlueFinalResNm']
#            self.wavelen = numpy.arange( xcal['SpecGlueStartWlNm'], 
#                                        xcal['SpecGlueEndWlNm']+step, step=step )
#            if len(self.wavelen) != len(self.lum):
#                self.wavelen = numpy.arange( xcal['SpecGlueStartWlNm'], 
#                                        xcal['SpecGlueEndWlNm'], step=step )
#                if len(self.wavelen) != len(self.lum):
#                    raise NameError('Wavelength calibration wrong. See get_spectrum \
#                                     in piUtils module.')
#        else:
#            if xcal['polynom_order'] != '\x02':
#                """
#                This is hex for "2", i.e. second order. I unpacked it as
#                a string rather than an int. Whatever...
#                """
#                raise NameError('"get_spectrum" in the winspec/piUtils module currently only handles second-order calibration.')
#                print '"get_spectrum" in the winspec/piUtils module currently only handles second-order calibration.'
#            pcoeffs = xcal['polynom_coeff']
#            print pcoeffs
#            p = numpy.array([ pcoeffs [2], pcoeffs [1], pcoeffs [0] ])
#            self.wavelen = numpy.polyval( p, xrange( 1, 1+len(self.lum) ) )

    def read_txt( self ):
        """
        Shanying 2/22/2012, reads simple text files with only 2 columns: wavelength and intensity
        """
        data = pylab.loadtxt(self.fname)
        self.wavelen = data[:,0]
        self.lum = data[:,1]
        self.flatfield_corrected = False
        self.background_corrected = False
        
    def remove_cosmic_rays( self, threshold=3.0, interpolate=False ):
        """
        This assumes that you've already loaded the file and have self.lum
        and self.wavelen.

        Remove spikes larger than
        threshold*(standard deviation of diffs in datay).
        (So higher threshold is less aggressive at removing points.)
        interpolate=False (default) just removes the points;
        interpolate=True will just make them equal to the point to the left 
        (if you need to keep array length constant, say).
        """
        try:
            self.lum
            self.wavelen
        except (AttributeError, NameError):
            return

        d = numpy.diff(self.lum)
        spikes = pylab.find(abs(d)>threshold*numpy.std(d)) 
        if ( len(spikes)>0.1*len(d) ): 
            print 'Did not remove spikes because it wanted to remove too many.'
        spikes=numpy.delete( spikes,pylab.find(numpy.diff(spikes)==1)+1 ) # if spike is one point, don't delete point after as well.
        if interpolate==False:
            self.wavelen = numpy.delete(self.wavelen,spikes+1)
            self.lum = numpy.delete(self.lum,spikes+1)
        else:
            # don't actually 'interpolate', just set it equal to the previous point
            # (actual interpolation could get messy in the case of a two-point spike, for example)
            for i in spikes+1:
                self.lum[i] = self.lum[i-3]
        if (numpy.any(numpy.abs(numpy.diff(self.lum))>threshold*numpy.std(d))):
            self.remove_cosmic_rays( threshold, interpolate )


    def leave_linear_background( self ):
        """ 
        A method to undo remove_linear_background(while_fitting=True).
        Sets the following variables:
        self._slope_remove  = False
        self._slope_npoints = None
        """
        self._slope_remove  = False
        self._slope_npoints = None


    def remove_linear_background( self, npoints=None, while_fitting=False ):
        """
        removes linear background from spectrum by fitting a line
        to the npoints on each end, then subtracting.
        If npoints is None (default) then the full range is used.
        Then, if data.min() is negative, the data is shifted up so
        that it's all positive.
        
        This is mostly useful when plotting colormaps of spectra
        versus time or position, to somewhat equalize the colormap.
        
        while_fitting=True means that the data itself isn't altered
        but polyval is used to subtract off a linear background to
        just the section being fitted (which is then added back onto
        the fit).
        """
        if while_fitting:
            self._slope_remove = True
            self._slope_npoints = npoints
            return None
            
        if npoints is None:
            fit = polyfit( self.wavelen, self.lum, 1 )
        else:
            x = numpy.zeros( 2*npoints )
            x[:npoints] = self.wavelen[:npoints]
            x[npoints:] = self.wavelen[-npoints:]
            
            y = numpy.zeros( 2*npoints )
            y[:npoints] = self.lum[:npoints]
            y[npoints:] = self.lum[-npoints:]
            
            fit = numpy.polyfit( x, y, 1 )

        self.lum -= numpy.polyval( fit, self.wavelen )

        if self.lum.min() < 0:
            self.lum -= self.lum.min()
            
    def reset_lum( self ):
        """ undo any normalization, background correction, or counts/second conversion, etc."""
        self.lum = self.raw_lum.copy()
        self._counts_per_second = False

    def save_ascii( self, fname ):
        g=numpy.zeros([len(self.lum),2])
        g[:,0]=self.wavelen
        g[:,1]=self.lum
        numpy.savetxt( fname, g )
        
    def set_axes( self, axes ):
        """ this allows you to plot the data to a particular axes
        """
        self.ax = axes

    def set_laser( self, laser ):
        """ Set the wavelength (in nm) of the exciting laser (useful for plotting with as_raman=True)
            Example usage for 532nm laser: self.set_laser( 532.0 )
        """
        self.laser = laser
        
################################################### 
################################################### 
################################################### 
"""
Following functions are not in the Spectrum class
"""


def get_spectrum( spefile, cts_per_sec=False ):
    """ 
    Read in an SPE file and return the spectrum as a numpy array.
    usage:

        wavelength, luminescence = get_spectrum( spe_filename )
    
    If you pass it the additional argument

       cts_per_sec=True

    then it will return the spectrum in counts per second rather than just counts (the default).
    """
    spe_struct, fileinfo = read_spe( spefile )
    luminescence = numpy.transpose( spe_struct['data'][0] )

    # the luminescence is stored as a 2D matrix (capable of holding each pixel
    # in the CCD array), even though it's only 1D of data (assuming you're in
    # 'spectrum' mode and thus that the columns of the CCD were integrated)
    # Here we flatten it to a 1D array:        
    luminescence = luminescence.flatten()
        
    if cts_per_sec:
        """Normalize to counts per second."""
        luminescence /= spe_struct['EXPOSURE']

    xcal = spe_struct['XCALIB']

    wavelength = numpy.arange(0, len(luminescence))


# Calibration is commented out # ximeg
#    if xcal['SpecGlueFlag']:
#        # spectrum was 'glued' from spectra at different grating angles
#        step=xcal['SpecGlueFinalResNm']
#        wavelength = numpy.arange( xcal['SpecGlueStartWlNm'], 
#                                    xcal['SpecGlueEndWlNm']+step, step=step )
#
#        if len(wavelength) != len(luminescence):
#            wavelength = numpy.arange( xcal['SpecGlueStartWlNm'], 
#                                    xcal['SpecGlueEndWlNm'], step=step )

#            if len(wavelength) != len(luminescence):
#                raise NameError('Wavelength calibration wrong. See get_spectrum in piUtils module.')

#    else:
#        if xcal['polynom_order'] != '\x02':
#            """
#            This is hex for "2", i.e. second order. I unpacked it as a string
#            rather than an int. Whatever...
#            """
#            raise NameError('"get_spectrum" in the winspec/piUtils module currently only handles second-order calibration.')

#        pcoeffs = xcal['polynom_coeff']
#        p = numpy.array([ pcoeffs [2], pcoeffs [1], pcoeffs [0] ])
#        wavelength = numpy.polyval( p, xrange( 1, 1+len(luminescence) ) )

    return wavelength, luminescence


def read_spe(spefilename, verbose=False):
    """ 
    Read a binary PI SPE file into a python dictionary

    Inputs:

        spefilename --  string specifying the name of the SPE file to be read
        verbose     --  boolean print debug statements (True) or not (False)

        Outputs
        spedict     
        
            python dictionary containing header and data information
            from the SPE file
            Content of the dictionary is:
            spedict = {'data':[],    # a list of 2D numpy arrays, one per image
            'IGAIN':pimaxGain,
            'EXPOSURE':exp_sec,
            'SPEFNAME':spefilename,
            'OBSDATE':date,
            'CHIPTEMP':detectorTemperature
            }

    I use the struct module to unpack the binary SPE data.
    Some useful formats for struct.unpack_from() include:
    fmt   c type          python
    c     char            string of length 1
    s     char[]          string (Ns is a string N characters long)
    h     short           integer 
    H     unsigned short  integer
    l     long            integer
    f     float           float
    d     double          float

    The SPE file defines new c types including:
        BYTE  = unsigned char
        WORD  = unsigned short
        DWORD = unsigned long


    Example usage:
    Given an SPE file named test.SPE, you can read the SPE data into
    a python dictionary named spedict with the following:
    >>> import piUtils
    >>> spedict = piUtils.readSpe('test.SPE')
    """
  
    # open SPE file as binary input
    spe = open(spefilename, "rb")
    
    # Header length is a fixed number
    nBytesInHeader = 4100

    # Read the entire header
    header = spe.read(nBytesInHeader)
    
    # version of WinView used
    swversion = struct.unpack_from("16s", header, offset=688)[0]
    
    # version of header used
    # Eventually, need to adjust the header unpacking
    # based on the headerVersion.  
    headerVersion = struct.unpack_from("f", header, offset=1992)[0]
  
    # which camera controller was used?
    controllerVersion = struct.unpack_from("h", header, offset=0)[0]

    fileinfo =  "swVersion         = %s\n" % swversion
    fileinfo += "headerVersion     = %.3f\n" % headerVersion
    fileinfo += "controllerVersion = %s\n" % controllerVersion
    
    # Date of the observation
    # (format is DDMONYYYY  e.g. 27Jan2009)
    date = struct.unpack_from("9s", header, offset=20)[0]
    
    # Exposure time (float)
    exp_sec = struct.unpack_from("f", header, offset=10)[0]
    
    # Intensifier gain
    pimaxGain = struct.unpack_from("h", header, offset=148)[0]

    # Not sure which "gain" this is
    gain = struct.unpack_from("H", header, offset=198)[0]
    
    # Data type (0=float, 1=long integer, 2=integer, 3=unsigned int)
    data_type = struct.unpack_from("h", header, offset=108)[0]

    comments = struct.unpack_from("400s", header, offset=200)[0]
    comments = comments.replace("\x00", " ").strip()
    while comments.find("  ") != -1:
        comments = comments.replace("  ", " ")
    comments = "; ".join(comments.splitlines())
    comments = "\n".join(textwrap.wrap(comments, width=28))

    # CCD Chip Temperature (Degrees C)
    detectorTemperature = struct.unpack_from("f", header, offset=36)[0]

    # The following get read but are not used
    # (this part is only lightly tested...)
    analogGain = struct.unpack_from("h", header, offset=4092)[0]
    noscan = struct.unpack_from("h", header, offset=34)[0]
    pimaxUsed = struct.unpack_from("h", header, offset=144)[0]
    pimaxMode = struct.unpack_from("h", header, offset=146)[0]

    ########### here's from Kasey
    #int avgexp 2 number of accumulations per scan (why don't they call this "accumulations"?)
#TODO: this isn't actually accumulations, so fix it...    
    accumulations = struct.unpack_from("h", header, offset=668)[0]
    if accumulations == -1:
        # if > 32767, set to -1 and 
        # see lavgexp below (668) 
        #accumulations = struct.unpack_from("l", header, offset=668)[0]
        # or should it be DWORD, NumExpAccums (1422): Number of Time experiment accumulated        
        accumulations = struct.unpack_from("l", header, offset=1422)[0]
        
    """Start of X Calibration Structure (although I added things to it that I
       thought were relevant, like the center wavelength..."""
    xcalib = {}
    
    #SHORT SpecAutoSpectroMode 70 T/F Spectrograph Used
    xcalib['SpecAutoSpectroMode'] = bool( struct.unpack_from("h", header, offset=70)[0] )

    #float SpecCenterWlNm # 72 Center Wavelength in Nm
    xcalib['SpecCenterWlNm'] = struct.unpack_from("f", header, offset=72)[0]
    
    #SHORT SpecGlueFlag 76 T/F File is Glued
    xcalib['SpecGlueFlag'] = bool( struct.unpack_from("h", header, offset=76)[0] )

    #float SpecGlueStartWlNm 78 Starting Wavelength in Nm
    xcalib['SpecGlueStartWlNm'] = struct.unpack_from("f", header, offset=78)[0]

    #float SpecGlueEndWlNm 82 Starting Wavelength in Nm
    xcalib['SpecGlueEndWlNm'] = struct.unpack_from("f", header, offset=82)[0]

    #float SpecGlueMinOvrlpNm 86 Minimum Overlap in Nm
    xcalib['SpecGlueMinOvrlpNm'] = struct.unpack_from("f", header, offset=86)[0]

    #float SpecGlueFinalResNm 90 Final Resolution in Nm
    xcalib['SpecGlueFinalResNm'] = struct.unpack_from("f", header, offset=90)[0]

    #  short   BackGrndApplied              150  1 if background subtraction done
    xcalib['BackgroundApplied'] = struct.unpack_from("h", header, offset=150)[0]
    BackgroundApplied=False
    if xcalib['BackgroundApplied']==1: BackgroundApplied=True

    #  float   SpecGrooves                  650  Spectrograph Grating Grooves
    xcalib['SpecGrooves'] = struct.unpack_from("f", header, offset=650)[0]

    #  short   flatFieldApplied             706  1 if flat field was applied.
    xcalib['flatFieldApplied'] = struct.unpack_from("h", header, offset=706)[0]
    flatFieldApplied=False
    if xcalib['flatFieldApplied']==1: flatFieldApplied=True
    
    #double offset # 3000 offset for absolute data scaling */
    xcalib['offset'] = struct.unpack_from("d", header, offset=3000)[0]

    #double factor # 3008 factor for absolute data scaling */
    xcalib['factor'] = struct.unpack_from("d", header, offset=3008)[0]
    
    #char current_unit # 3016 selected scaling unit */
    xcalib['current_unit'] = struct.unpack_from("c", header, offset=3016)[0]

    #char reserved1 # 3017 reserved */
    xcalib['reserved1'] = struct.unpack_from("c", header, offset=3017)[0]

    #char string[40] # 3018 special string for scaling */
    xcalib['string'] = struct.unpack_from("40c", header, offset=3018)
    
    #char reserved2[40] # 3058 reserved */
    xcalib['reserved2'] = struct.unpack_from("40c", header, offset=3058)

    #char calib_valid # 3098 flag if calibration is valid */
    xcalib['calib_valid'] = struct.unpack_from("c", header, offset=3098)[0]

    #char input_unit # 3099 current input units for */
    xcalib['input_unit'] = struct.unpack_from("c", header, offset=3099)[0]
    """/* "calib_value" */"""

    #char polynom_unit # 3100 linear UNIT and used */
    xcalib['polynom_unit'] = struct.unpack_from("c", header, offset=3100)[0]
    """/* in the "polynom_coeff" */"""

    #char polynom_order # 3101 ORDER of calibration POLYNOM */
    xcalib['polynom_order'] = struct.unpack_from("c", header, offset=3101)[0]

    #char calib_count # 3102 valid calibration data pairs */
    xcalib['calib_count'] = struct.unpack_from("c", header, offset=3102)[0]

    #double pixel_position[10];/* 3103 pixel pos. of calibration data */
    xcalib['pixel_position'] = struct.unpack_from("10d", header, offset=3103)

    #double calib_value[10] # 3183 calibration VALUE at above pos */
    xcalib['calib_value'] = struct.unpack_from("10d", header, offset=3183)

    #double polynom_coeff[6] # 3263 polynom COEFFICIENTS */
    xcalib['polynom_coeff'] = struct.unpack_from("6d", header, offset=3263)

    #double laser_position # 3311 laser wavenumber for relativ WN */
    xcalib['laser_position'] = struct.unpack_from("d", header, offset=3311)[0]

    #char reserved3 # 3319 reserved */
    xcalib['reserved3'] = struct.unpack_from("c", header, offset=3319)[0]

    #unsigned char new_calib_flag # 3320 If set to 200, valid label below */
    #xcalib['calib_value'] = struct.unpack_from("BYTE", header, offset=3320)[0] # how to do this?

    #char calib_label[81] # 3321 Calibration label (NULL term'd) */
    xcalib['calib_label'] = struct.unpack_from("81c", header, offset=3321)

    #char expansion[87] # 3402 Calibration Expansion area */
    xcalib['expansion'] = struct.unpack_from("87c", header, offset=3402)
    ########### end of Kasey's addition
    fileinfo += "date              = %s\n" % date
    fileinfo += "exposure_sec      = %s\n" % exp_sec
    fileinfo += "pimaxGain         = %s\n" % pimaxGain
    fileinfo += "gain              = %s\n" % gain
    fileinfo += "dataType          = %s\n" % data_type
    fileinfo += "analogGain        = %s\n" % analogGain
    fileinfo += "noscan            = %s\n" % noscan
    fileinfo += "detectorTemp_C    = %.1f\n" % detectorTemperature
    fileinfo += "pimaxUsed         = %s\n" % pimaxUsed


    # Determine the data type format string for
    # upcoming struct.unpack_from() calls
    if data_type == 0:
        # float (4 bytes)
        dataTypeStr = "f"  #untested
        bytesPerPixel = 4
        dtype = "float32"
    elif data_type == 1:
        # long (4 bytes)
        dataTypeStr = "l"  #untested
        bytesPerPixel = 4
        dtype = "int32"
    elif data_type == 2:
        # short (2 bytes)
        dataTypeStr = "h"  #untested
        bytesPerPixel = 2
        dtype = "int32"
    elif data_type == 3:  
        # unsigned short (2 bytes)
        dataTypeStr = "H"  # 16 bits in python on intel mac
        bytesPerPixel = 2
        dtype = "int32"  # for numpy.array().
        # other options include:
        # IntN, UintN, where N = 8,16,32 or 64
        # and Float32, Float64, Complex64, Complex128
        # but need to verify that pyfits._ImageBaseHDU.ImgCode cna handle it
        # right now, ImgCode must be float32, float64, int16, int32, int64 or uint8
    else:
        print "unknown data type"
        print "returning..."
        sys.exit()
  
    # Number of pixels on x-axis and y-axis
    nx = struct.unpack_from("H", header, offset=42)[0]
    ny = struct.unpack_from("H", header, offset=656)[0]
    
    # Number of image frames in this SPE file
    nframes = struct.unpack_from("l", header, offset=1446)[0]

    fileinfo += "nx, ny, nframes   = %i, %i, %i\n" % (nx, ny, nframes)
    
    npixels = nx*ny
    npixStr = str(npixels)
    fmtStr  = npixStr + dataTypeStr

    fileinfo += "fmtStr            = %s\n" % fmtStr
    
    # How many bytes per image?
    nbytesPerFrame = npixels*bytesPerPixel

    fileinfo += "nbytesPerFrame    = %i\n" % nbytesPerFrame

    # Create a dictionary that holds some header information
    # and contains a placeholder for the image data
    spedict = {'data':[],    # can have more than one image frame per SPE file
                'IGAIN':pimaxGain,
                'GAIN':gain,
                'EXPOSURE':exp_sec,
                'SPEFNAME':spefilename,
                'OBSDATE':date,
                'CHIPTEMP':detectorTemperature,
                'COMMENTS':comments,
                'XCALIB':xcalib,
                'ACCUMULATIONS':accumulations,
                'FLATFIELD':flatFieldApplied,
                'BACKGROUND':BackgroundApplied
                }
    
    # Now read in the image data
    # Loop over each image frame in the image

    fileinfo += 'comments          = "%s"\n' % comments
    if verbose:
        print fileinfo + "\n\n"
        print "Reading image frames number ",
    for ii in range(nframes):
        iistr = str(ii)
        data = spe.read(nbytesPerFrame)
        if verbose:
            print ii," ",
    
        # read pixel values into a 1-D numpy array. the "=" forces it to use
        # standard python datatype size (4bytes for 'l') rather than native
        # (which on 64bit is 8bytes for 'l', for example).
        # See http://docs.python.org/library/struct.html
        dataArr = numpy.array(struct.unpack_from("="+fmtStr, data, offset=0),
                            dtype=dtype)

        # Resize array to nx by ny pixels
        # notice order... (y,x)
        dataArr.resize((ny, nx))
        #print dataArr.shape

        # Push this image frame data onto the end of the list of images
        # but first cast the datatype to float (if it's not already)
        # this isn't necessary, but shouldn't hurt and could save me
        # from doing integer math when i really meant floating-point...
        spedict['data'].append( dataArr.astype(float) )

    if verbose:
        print ""
  
    return spedict, fileinfo.replace("\x00", "")
  

###############################################################################
###############################################################################
####        Description of the header structure used to create piUtils      ###
###############################################################################
###############################################################################
#
#                                  WINHEAD.TXT
#
#                            $Date: 3/23/04 11:36 $
#
#                Header Structure For WinView/WinSpec (WINX) Files
#
#  The current data file used for WINX files consists of a 4100 (1004 Hex)
#  byte header followed by the data.
#
#  Beginning with Version 2.5, many more items were added to the header to
#  make it a complete as possible record of the data collection.  This includes
#  spectrograph and pulser information.  Much of these additions were accomplished
#  by recycling old information which had not been used in many versions.
#  All data files created under previous 2.x versions of WinView/WinSpec CAN
#  still be read correctly.  HOWEVER, files created under the new versions
#  (2.5 and higher) CANNOT be read by previous versions of WinView/WinSpec
#  OR by the CSMA software package.
#
#
#            ***************************************************
#
#                                    Decimal Byte
#                                       Offset
#                                    -----------
#  short   ControllerVersion              0  Hardware Version
#  short   LogicOutput                    2  Definition of Output BNC
#  WORD    AmpHiCapLowNoise               4  Amp Switching Mode
#  WORD    xDimDet                        6  Detector x dimension of chip.
#  short   mode                           8  timing mode
#  float   exp_sec                       10  alternitive exposure, in sec.
#  short   VChipXdim                     14  Virtual Chip X dim
#  short   VChipYdim                     16  Virtual Chip Y dim
#  WORD    yDimDet                       18  y dimension of CCD or detector.
#  char    date[DATEMAX]                 20  date
#  short   VirtualChipFlag               30  On/Off
#  char    Spare_1[2]                    32
#  short   noscan                        34  Old number of scans - should always be -1
#  float   DetTemperature                36  Detector Temperature Set
#  short   DetType                       40  CCD/DiodeArray type
#  WORD    xdim                          42  actual # of pixels on x axis
#  short   stdiode                       44  trigger diode
#  float   DelayTime                     46  Used with Async Mode
#  WORD    ShutterControl                50  Normal, Disabled Open, Disabled Closed
#  short   AbsorbLive                    52  On/Off
#  WORD    AbsorbMode                    54  Reference Strip or File
#  short   CanDoVirtualChipFlag          56  T/F Cont/Chip able to do Virtual Chip
#  short   ThresholdMinLive              58  On/Off
#  float   ThresholdMinVal               60  Threshold Minimum Value
#  short   ThresholdMaxLive              64  On/Off
#  float   ThresholdMaxVal               66  Threshold Maximum Value
#  short   SpecAutoSpectroMode           70  T/F Spectrograph Used
#  float   SpecCenterWlNm                72  Center Wavelength in Nm
#  short   SpecGlueFlag                  76  T/F File is Glued
#  float   SpecGlueStartWlNm             78  Starting Wavelength in Nm
#  float   SpecGlueEndWlNm               82  Starting Wavelength in Nm
#  float   SpecGlueMinOvrlpNm            86  Minimum Overlap in Nm
#  float   SpecGlueFinalResNm            90  Final Resolution in Nm
#  short   PulserType                    94  0=None, PG200=1, PTG=2, DG535=3
#  short   CustomChipFlag                96  T/F Custom Chip Used
#  short   XPrePixels                    98  Pre Pixels in X direction
#  short   XPostPixels                  100  Post Pixels in X direction
#  short   YPrePixels                   102  Pre Pixels in Y direction 
#  short   YPostPixels                  104  Post Pixels in Y direction
#  short   asynen                       106  asynchron enable flag  0 = off
#  short   datatype                     108  experiment datatype
#                                             0 =   float (4 bytes)
#                                             1 =   long (4 bytes)
#                                             2 =   short (2 bytes)
#                                             3 =   unsigned short (2 bytes)
#  short   PulserMode                   110  Repetitive/Sequential
#  WORD    PulserOnChipAccums           112  Num PTG On-Chip Accums
#  DWORD   PulserRepeatExp              114  Num Exp Repeats (Pulser SW Accum)
#  float   PulseRepWidth                118  Width Value for Repetitive pulse (usec)
#  float   PulseRepDelay                122  Width Value for Repetitive pulse (usec)
#  float   PulseSeqStartWidth           126  Start Width for Sequential pulse (usec)
#  float   PulseSeqEndWidth             130  End Width for Sequential pulse (usec)
#  float   PulseSeqStartDelay           134  Start Delay for Sequential pulse (usec)
#  float   PulseSeqEndDelay             138  End Delay for Sequential pulse (usec)
#  short   PulseSeqIncMode              142  Increments: 1=Fixed, 2=Exponential
#  short   PImaxUsed                    144  PI-Max type controller flag
#  short   PImaxMode                    146  PI-Max mode
#  short   PImaxGain                    148  PI-Max Gain
#  short   BackGrndApplied              150  1 if background subtraction done
#  short   PImax2nsBrdUsed              152  T/F PI-Max 2ns Board Used
#  WORD    minblk                       154  min. # of strips per skips
#  WORD    numminblk                    156  # of min-blocks before geo skps
#  short   SpecMirrorLocation[2]        158  Spectro Mirror Location, 0=Not Present
#  short   SpecSlitLocation[4]          162  Spectro Slit Location, 0=Not Present
#  short   CustomTimingFlag             170  T/F Custom Timing Used
#  char    ExperimentTimeLocal[TIMEMAX] 172  Experiment Local Time as hhmmss\0
#  char    ExperimentTimeUTC[TIMEMAX]   179  Experiment UTC Time as hhmmss\0
#  short   ExposUnits                   186  User Units for Exposure
#  WORD    ADCoffset                    188  ADC offset
#  WORD    ADCrate                      190  ADC rate
#  WORD    ADCtype                      192  ADC type
#  WORD    ADCresolution                194  ADC resolution
#  WORD    ADCbitAdjust                 196  ADC bit adjust
#  WORD    gain                         198  gain
#  char    Comments[5][COMMENTMAX]      200  File Comments
#  WORD    geometric                    600  geometric ops: rotate 0x01,
#                                             reverse 0x02, flip 0x04
#  char    xlabel[LABELMAX]             602  intensity display string
#  WORD    cleans                       618  cleans
#  WORD    NumSkpPerCln                 620  number of skips per clean.
#  short   SpecMirrorPos[2]             622  Spectrograph Mirror Positions
#  float   SpecSlitPos[4]               626  Spectrograph Slit Positions
#  short   AutoCleansActive             642  T/F
#  short   UseContCleansInst            644  T/F
#  short   AbsorbStripNum               646  Absorbance Strip Number
#  short   SpecSlitPosUnits             648  Spectrograph Slit Position Units
#  float   SpecGrooves                  650  Spectrograph Grating Grooves
#  short   srccmp                       654  number of source comp. diodes
#  WORD    ydim                         656  y dimension of raw data.
#  short   scramble                     658  0=scrambled,1=unscrambled
#  short   ContinuousCleansFlag         660  T/F Continuous Cleans Timing Option
#  short   ExternalTriggerFlag          662  T/F External Trigger Timing Option
#  long    lnoscan                      664  Number of scans (Early WinX)
#  long    lavgexp                      668  Number of Accumulations
#  float   ReadoutTime                  672  Experiment readout time
#  short   TriggeredModeFlag            676  T/F Triggered Timing Option
#  char    Spare_2[10]                  678  
#  char    sw_version[FILEVERMAX]       688  Version of SW creating this file
#  short   type                         704   1 = new120 (Type II)             
#                                             2 = old120 (Type I )           
#                                             3 = ST130                      
#                                             4 = ST121                      
#                                             5 = ST138                      
#                                             6 = DC131 (PentaMax)           
#                                             7 = ST133 (MicroMax/SpectroMax)
#                                             8 = ST135 (GPIB)               
#                                             9 = VICCD                      
#                                            10 = ST116 (GPIB)               
#                                            11 = OMA3 (GPIB)                
#                                            12 = OMA4                       
#  short   flatFieldApplied             706  1 if flat field was applied.
#  char    Spare_3[16]                  708  
#  short   kin_trig_mode                724  Kinetics Trigger Mode
#  char    dlabel[LABELMAX]             726  Data label.
#  char    Spare_4[436]                 742
#  char    PulseFileName[HDRNAMEMAX]   1178  Name of Pulser File with
#                                             Pulse Widths/Delays (for Z-Slice)
#  char    AbsorbFileName[HDRNAMEMAX]  1298 Name of Absorbance File (if File Mode)
#  DWORD   NumExpRepeats               1418  Number of Times experiment repeated
#  DWORD   NumExpAccums                1422  Number of Time experiment accumulated
#  short   YT_Flag                     1426  Set to 1 if this file contains YT data
#  float   clkspd_us                   1428  Vert Clock Speed in micro-sec
#  short   HWaccumFlag                 1432  set to 1 if accum done by Hardware.
#  short   StoreSync                   1434  set to 1 if store sync used
#  short   BlemishApplied              1436  set to 1 if blemish removal applied
#  short   CosmicApplied               1438  set to 1 if cosmic ray removal applied
#  short   CosmicType                  1440  if cosmic ray applied, this is type
#  float   CosmicThreshold             1442  Threshold of cosmic ray removal.  
#  long    NumFrames                   1446  number of frames in file.         
#  float   MaxIntensity                1450  max intensity of data (future)    
#  float   MinIntensity                1454  min intensity of data (future)    
#  char    ylabel[LABELMAX]            1458  y axis label.                     
#  WORD    ShutterType                 1474  shutter type.                     
#  float   shutterComp                 1476  shutter compensation time.        
#  WORD    readoutMode                 1480  readout mode, full,kinetics, etc  
#  WORD    WindowSize                  1482  window size for kinetics only.    
#  WORD    clkspd                      1484  clock speed for kinetics & frame transfer
#  WORD    interface_type              1486  computer interface                
#                                             (isa-taxi, pci, eisa, etc.)             
#  short   NumROIsInExperiment         1488  May be more than the 10 allowed in
#                                             this header (if 0, assume 1)            
#  char    Spare_5[16]                 1490                                    
#  WORD    controllerNum               1506  if multiple controller system will
#                                             have controller number data came from.  
#                                             this is a future item.                  
#  WORD    SWmade                      1508  Which software package created this file 
#  short   NumROI                      1510  number of ROIs used. if 0 assume 1.      
#
#                                      
#-------------------------------------------------------------------------------
#
#                        ROI entries   (1512 - 1631)
#
#  struct ROIinfo                           
#  {                                                   
#    WORD  startx                            left x start value.               
#    WORD  endx                              right x value.                    
#    WORD  groupx                            amount x is binned/grouped in hw. 
#    WORD  starty                            top y start value.                
#    WORD  endy                              bottom y value.                   
#    WORD  groupy                            amount y is binned/grouped in hw. 
#  } ROIinfoblk[ROIMAX]                   
#                                            ROI Starting Offsets:     
#                                                 
#                                              ROI  1  =  1512     
#                                              ROI  2  =  1524     
#                                              ROI  3  =  1536     
#                                              ROI  4  =  1548     
#                                              ROI  5  =  1560     
#                                              ROI  6  =  1572     
#                                              ROI  7  =  1584     
#                                              ROI  8  =  1596     
#                                              ROI  9  =  1608     
#                                              ROI 10  =  1620
#
#-------------------------------------------------------------------------------
#
#  char    FlatField[HDRNAMEMAX]       1632  Flat field file name.       
#  char    background[HDRNAMEMAX]      1752  background sub. file name.  
#  char    blemish[HDRNAMEMAX]         1872  blemish file name.          
#  float   file_header_ver             1992  version of this file header 
#  char    YT_Info[1000]               1996-2995  Reserved for YT information
#  long    WinView_id                  2996  == 0x01234567L if file created by WinX
#
#-------------------------------------------------------------------------------
#
#                        START OF X CALIBRATION STRUCTURE (3000 - 3488)
#
#  double  offset                      3000  offset for absolute data scaling
#  double  factor                      3008  factor for absolute data scaling
#  char    current_unit                3016  selected scaling unit           
#  char    reserved1                   3017  reserved                        
#  char    string[40]                  3018  special string for scaling      
#  char    reserved2[40]               3058  reserved                        
#  char    calib_valid                 3098  flag if calibration is valid    
#  char    input_unit                  3099  current input units for         
#                                            "calib_value"                  
#  char    polynom_unit                3100  linear UNIT and used            
#                                            in the "polynom_coeff"         
#  char    polynom_order               3101  ORDER of calibration POLYNOM    
#  char    calib_count                 3102  valid calibration data pairs    
#  double  pixel_position[10]          3103  pixel pos. of calibration data  
#  double  calib_value[10]             3183  calibration VALUE at above pos  
#  double  polynom_coeff[6]            3263  polynom COEFFICIENTS            
#  double  laser_position              3311  laser wavenumber for relativ WN 
#  char    reserved3                   3319  reserved                        
#  BYTE    new_calib_flag              3320  If set to 200, valid label below
#  char    calib_label[81]             3321  Calibration label (NULL term'd) 
#  char    expansion[87]               3402  Calibration Expansion area      
#
#-------------------------------------------------------------------------------
#
#                        START OF Y CALIBRATION STRUCTURE (3489 - 3977)
#
#  double  offset                      3489  offset for absolute data scaling
#  double  factor                      3497  factor for absolute data scaling
#  char    current_unit                3505  selected scaling unit           
#  char    reserved1                   3506  reserved                        
#  char    string[40]                  3507  special string for scaling      
#  char    reserved2[40]               3547  reserved                        
#  char    calib_valid                 3587  flag if calibration is valid    
#  char    input_unit                  3588  current input units for         
#                                            "calib_value"                   
#  char    polynom_unit                3589  linear UNIT and used            
#                                            in the "polynom_coeff"          
#  char    polynom_order               3590  ORDER of calibration POLYNOM    
#  char    calib_count                 3591  valid calibration data pairs    
#  double  pixel_position[10]          3592  pixel pos. of calibration data  
#  double  calib_value[10]             3672  calibration VALUE at above pos  
#  double  polynom_coeff[6]            3752  polynom COEFFICIENTS            
#  double  laser_position              3800  laser wavenumber for relativ WN 
#  char    reserved3                   3808  reserved                        
#  BYTE    new_calib_flag              3809  If set to 200, valid label below
#  char    calib_label[81]             3810  Calibration label (NULL term'd) 
#  char    expansion[87]               3891  Calibration Expansion area      
#
#                         END OF CALIBRATION STRUCTURES
#
#-------------------------------------------------------------------------------
#
#  char    Istring[40]                 3978  special intensity scaling string
#  char    Spare_6[25]                 4018  
#  BYTE    SpecType                    4043  spectrometer type (acton, spex, etc.)
#  BYTE    SpecModel                   4044  spectrometer model (type dependent)
#  BYTE    PulseBurstUsed              4045  pulser burst mode on/off
#  DWORD   PulseBurstCount             4046  pulser triggers per burst
#  double  ulseBurstPeriod             4050  pulser burst period (in usec)
#  BYTE    PulseBracketUsed            4058  pulser bracket pulsing on/off
#  BYTE    PulseBracketType            4059  pulser bracket pulsing type
#  double  PulseTimeConstFast          4060  pulser slow exponential time constant (in usec)
#  double  PulseAmplitudeFast          4068  pulser fast exponential amplitude constant
#  double  PulseTimeConstSlow          4076  pulser slow exponential time constant (in usec)
#  double  PulseAmplitudeSlow          4084  pulser slow exponential amplitude constant
#  short   AnalogGain;                 4092  analog gain
#  short   AvGainUsed                  4094  avalanche gain was used
#  short   AvGain                      4096  avalanche gain value
#  short   lastvalue                   4098  Always the LAST value in the header
#
#                         END OF HEADER
#
#-------------------------------------------------------------------------------
#
#                                      4100  Start of Data
#
#
#
#        ***************************** E.O.F.*****************************
#
#
#
#
#  Definitions of array sizes:
#  ---------------------------
#
#    HDRNAMEMAX  = 120     Max char str length for file name
#    USERINFOMAX = 1000    User information space
#    COMMENTMAX  = 80      User comment string max length (5 comments)
#    LABELMAX    = 16      Label string max length
#    FILEVERMAX  = 16      File version string max length
#    DATEMAX     = 10      String length of file creation date string as ddmmmyyyy\0
#    ROIMAX      = 10      Max size of roi array of structures
#    TIMEMAX     = 7       Max time store as hhmmss\0
#
#
#
#  Custom Data Types used in the structure:
#  ----------------------------------------
#
#    BYTE = unsigned char
#    WORD = unsigned short
#    DWORD = unsigned long
#
#
#
#
#  READING DATA:
#  -------------
#
#    The data follows the header beginning at offset 4100.
#
#    Data is stored as sequential points.
#    
#    The X, Y and Frame dimensions are determined by the header.
#
#      The X dimension of the stored data is in "xdim" ( Offset 42  ).
#      The Y dimension of the stored data is in "ydim" ( Offset 656 ).
#      The number of frames of data stored is in "NumFrames" ( Offset 1446 ).
#
#    The size of a frame (in bytes) is:
#
#      One frame size = xdim x ydim x (datatype Offset 108)
#
###############################################################################
###############################################################################


###############################################################################
###############################################################################
#######################################
#### Lightly tested routines below ####
#######################################
###############################################################################
###############################################################################


#def speDictToFitsMultiExt(spedict, outfile=None, clobber=False, verbose=False):
#  """ Given an SPE file containing multiple exposures, create
#      a single, multi-extension FITS file with one image per HDU """
#    
#  # FITS output filename
#  if outfile == None:
#    fitsfileroot, junk = os.path.splitext(spedict['SPEFNAME'])
#    fitsFilename = fitsfileroot+".fits"
#  if verbose:
#    print "fitsFilename = ", fitsFilename
#    
#  hdrDesc = getHeaderDescriptions()
#
#  # Start a new HDU with image data
#  imglist = []
#  if verbose:
#    print "writing frame: ",
#  for frameNumber in range( len(spedict['data']) ):
#    if verbose:
#      print frameNumber, " ", 
#    if frameNumber == 0:
#      # Set up the primary image (the first frame)
#      # and header fields
#      hdu = pyfits.PrimaryHDU(spedict['data'][frameNumber])
#      for kk in hdrDesc.keys():
#        hdu.header.update(kk, spedict[kk], hdrDesc[kk])
#      imglist.append( hdu )
#    else:
#      imglist.append( pyfits.ImageHDU(spedict['data'][frameNumber]) )
#  if verbose:
#    print ""
#  
#  hdulist = pyfits.HDUList(imglist)
#      
#  hdulist.writeto(fitsFilename, clobber=clobber)
#
#  return fitsFilename

#def readHeaderCfg(cfgfile, verbose=False, comment='#'):
#  """ sample cfgfile is
#
#  # FITSHDRName  SPEHdrFmt offset %% description
#  IGAIN h 148 %% intensifier gain, 0-255
#
#  """
#  
#  descSep = '%%'
#  hdrCfg = {}
#  cfile = open(cfgfile, 'r')
#  for line in cfile:
#    line = line.strip()
#    if line.startswith(comment):
#      continue
#
#    hdr, description = line.split(descSep)
#    description = description.strip()
#
#    hdrName, hdrFmt, offset = hdr.split()
#    offset = int(offset)
#
#    hdrCfg[hdrName] = {'offset':offset, 'fmt':hdrFmt, 'description':description}
#
#  return hdrCfg

#def makeHeaderDict(spefilename, configfile, verbose=False):
#  """ external config file dictates what SPE header info
#  makes it into FITS file
#  """
#
#  cfgDict = readHeaderCfg(configfile)
#  print cfgDict
#
#  # open SPE file as binary input
#  spe = open(spefilename, "rb")
#  
#  # Header length is a fixed number
#  nBytesInHeader = 4100
#  # Read the entire header
#  header = spe.read(nBytesInHeader)
#
#  hdrDict = {}
#
#  for key in cfgDict.keys():
#    val = struct.unpack_from(cfgDict[key]['fmt'], header,
#                             offset=cfgDict[key]['offset'])[0]
#    hdrDict[key] = {'val':val, 'comment':cfgDict[key]['description']}
#
#  # then do the mandatory ones
#  mandatoryDict = {}
#  mandatoryDict['SPEDTYPE'] = {'offset':108, 'fmt':'h', 'description':'SPE file image data type'}
#  mandatoryDict['NX'] = {'offset':42, 'fmt':'H', 'description':'Number of x pixels'}
#  mandatoryDict['NY'] = {'offset':656, 'fmt':'H', 'description':'Number of y pixels'}
#  mandatoryDict['NFRAMES'] = {'offset':1446, 'fmt':'l', 'description':'Number of frames in original SPE file'}
#  
#  if verbose:
#    print hdrDict
#  spe.close()

#def getDataType(header, offset=108):
#
#  data_type = struct.unpack_from("h", header, offset=offset)[0]
#
#  # Determine the data type format string for
#  # upcoming struct.unpack_from() calls
#  if data_type == 0:
#    # float (4 bytes)
#    dataTypeStr = "f"  #untested
#    bytesPerPixel = 4
#  elif data_type == 1:
#    # long (4 bytes)
#    dataTypeStr = "l"  #untested
#    bytesPerPixel = 4
#  elif data_type == 2:
#    # short (2 bytes)
#    dataTypeStr = "h"  #untested
#    bytesPerPixel = 2
#  elif data_type == 3:  
#    # unsigned short (2 bytes)
#    dataTypeStr = "H"  # 16 bits in python on intel mac
#    bytesPerPixel = 2
#    dtype = "int32"  # for numpy.array().
#    # other options include:
#    # IntN, UintN, where N = 8,16,32 or 64
#    # and Float32, Float64, Complex64, Complex128
#    # but need to verify that pyfits._ImageBaseHDU.ImgCode cna handle it
#    # right now, ImgCode must be float32, float64, int16, int32, int64 or uint8
#  else:
#    print "unknown data type"
#    print "returning..."
#    return
#
#  return (dataTypeStr, bytesPerPixel, dtype)

