//
// RWLock.h
//
// $Id: //poco/1.3/Foundation/include/Poco/RWLock.h#1 $
//
// Library: Foundation
// Package: Threading
// Module:  RWLock
//
// Definition of the RWLock class.
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
//
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

#ifndef Foundation_RWLock_INCLUDED
#define Foundation_RWLock_INCLUDED

#include "Poco/Foundation.h"

#if defined(HAVE_POSIX) && !defined(ANDROID)
#include "Poco/RWLock_POSIX.h"
#else
#include "Poco/RWLock_Mutex.hpp"
#endif

namespace Poco {

class ScopedRWLock;

/**
 * A reader writer lock allows multiple concurrent
 * readers or one exclusive writer.
 */
class Foundation_API RWLock: private RWLockImpl
{
public:
  /** Type of lock */
	typedef ScopedRWLock ScopedLock;

	/** Creates the Reader/Writer lock. */
	RWLock();

	/** Destroys the Reader/Writer lock. */
	~RWLock();

	/**
	 * Acquires a read lock. If another thread currently holds a write lock,
	 *  waits until the write lock is released.
	 */
	void readLock();

	/**
	 * Tries to acquire a read lock. Immediately returns true if successful,
	 * or false if another thread currently holds a write lock.
	 */
	bool tryReadLock();

	/**
	 * Acquires a write lock. If one or more other threads currently hold
	 * locks, waits until all locks are released. The results are undefined
	 * if the same thread already holds a read or write lock
	 */
	void writeLock();

	/**
	 * Tries to acquire a write lock. Immediately returns true if successful,
	 *  or false if one or more other threads currently hold locks. The
	 *  result is undefined if the same thread already holds a read or write lock.
	 */
	bool tryWriteLock();

	/** Releases the read or write lock. */
	void unlock();

private:
	RWLock(const RWLock&);
	RWLock& operator = (const RWLock&);
};

/** A variant of ScopedLock for reader/writer locks. */
class Foundation_API ScopedRWLock
{
public:
/** 
 * Constructor
 * 
 * @param rwl Lock to use
 * @param write If true, locks read-write, otherwise read only
 * 
 * @return Scope object
 */
	ScopedRWLock(RWLock& rwl, bool write = false);
	~ScopedRWLock();

private:
	RWLock& _rwl;

	ScopedRWLock();
	ScopedRWLock(const ScopedRWLock&);
	ScopedRWLock& operator = (const ScopedRWLock&);
};


//
// inlines
//
inline void RWLock::readLock()
{
	readLockImpl();
}


inline bool RWLock::tryReadLock()
{
	return tryReadLockImpl();
}


inline void RWLock::writeLock()
{
	writeLockImpl();
}


inline bool RWLock::tryWriteLock()
{
	return tryWriteLockImpl();
}


inline void RWLock::unlock()
{
	unlockImpl();
}


inline ScopedRWLock::ScopedRWLock(RWLock& rwl, bool write): _rwl(rwl)
{
	if (write)
		_rwl.writeLock();
	else
		_rwl.readLock();
}


inline ScopedRWLock::~ScopedRWLock()
{
	_rwl.unlock();
}


} // namespace Poco


#endif // Foundation_RWLock_INCLUDED
