"""More things for working with coverage.py
"""

from UserList import UserList
import sys, os
from StringIO import StringIO
import lxml.etree as ET

__author__ = 'Drew Smathers'
__version__ = '$Revision$'[11:-2]

class AnnotationLine:
    def __init__(self, text, isexec=False, covered=False):
        self.text = text
        self.isexec = isexec
        self.covered = covered
        # hack
        if self.text and self.text[-1] == '\n':
            self.text = self.text[:-1]

class Annotation(UserList):
    pass

class AnnotationParser:
    """Parser for annotation files generated by coverage.
    """
    def parse(self, input):
        fd = input
        if not hasattr(input, 'read'):
            fd = open(input)
        annotation = Annotation()
        for line in input:
            if not line or line[0] not in ('>', '!'):
                annotation.append(AnnotationLine(line[2:], isexec=False))
            elif line[0] == '>':
                annotation.append(AnnotationLine(line[2:], isexec=True, covered=True))
            elif line[0] == '!':
                annotation.append(AnnotationLine(line[2:], isexec=True, covered=False))
            else:
                print >> sys.stderr, 'invalid annotation line: ' + line
        return annotation

def annotationToXML(annotation, tree=False):
    """Transform annotation object to XML string or ElementTree instance
    if tree is True.

    @param annotation: Annotation instance
    @param tree: set to true to return ElementTree instance.
    """
    etree = _annotationToXML(annotation)
    if tree:
        return etree
    buffer = StringIO()
    etree.write(buffer)
    return buffer.getvalue()

def _annotationToXML(annotation):
    root = ET.Element('coverageAnnotation')
    for line in annotation:
        aline = ET.SubElement(root, 'line')
        aline.attrib['executable'] = str(line.isexec).lower()
        aline.attrib['covered'] = str(line.covered).lower()
        aline.text = line.text
    return ET.ElementTree(root)
    

def annotationToHTML(annotation, xml=None, xslt=None, tree=False):
    """Transform annotation object to HTML string or ElementTree instance
    if tree is True.

    @param annotation: Annotation instance
    @param xslt: xslt source for transformation
    @param tree: set to true to return ElementTree instance.
    """
    style = xslt or _XSLT
    if not hasattr(style, 'read'):
        style = StringIO(style)
    style = ET.parse(style)
    etree = _annotationToXML(annotation)
    html = etree.xslt(style)
    if tree:
        return html
    buffer = StringIO()
    html.write(buffer)
    return buffer.getvalue()

###########################################
# Coverage reports
###########################################

class CoverageReport(UserList):
    package_name = None
    summary = None

class CoverageReportEntry:

    def __init__(self, modname, statements, executed, coverage, missing):
        self.modname = modname
        self.statements = statements
        self.executed = executed
        self.coverage = coverage
        self.missing = missing

class CoverageReportParser:

    def parse(self, input):
        fd = input
        if not hasattr(input, 'read'):
            fd = open(input)
        report = CoverageReport()
        for line in fd.readlines()[2:]:
            tokens = line.split()
            try:
                modname, stmts, execd, coverage = tokens[:4]
                perc = coverage[:-1]
                _stmts, _execd = int(stmts), int(execd)
            except Exception, e:
                continue
            if modname == 'TOTAL':
                report.summary = CoverageReportEntry(modname, stmts, execd, coverage, None)
                break
            modname = modname.replace(os.path.sep, '.')
            missed = [ tk.replace(',','') for tk in tokens[4:] ]
            report.append(CoverageReportEntry(modname, stmts, execd, coverage, missed))
        return report


def reportToXML(report, tree=False):
    """Transform report object to XML representation as string or ElementTree
    instance if tree arg is set to True.

    @param report: report instance
    @param tree: set to true to return ElementTree instance
    """
    etree = _reportToXML(report)
    if tree:
        return etree
    buffer = StringIO()
    etree.write(buffer)
    return buffer.getvalue()

def _reportToXML(report):
    root = ET.Element('coverage-report')
    elm = ET.SubElement(root, 'summary')
    attr = elm.attrib
    attr['statements'] = report.summary.statements
    attr['executed'] = report.summary.executed
    attr['coverage'] = report.summary.coverage
    for entry in report:
        elm = ET.SubElement(root, 'module')
        attr = elm.attrib
        attr['name'] = entry.modname
        attr['statements'] = entry.statements
        attr['coverage'] = entry.coverage
        attr['executed'] = entry.executed
        melm = ET.SubElement(elm, 'missing-ranges')
        for missed in entry.missing:
            start_end = missed.split('-')
            if len(start_end) == 2:
                start, end = start_end
            else:
                start = end = start_end[0]
            relm = ET.SubElement(melm, 'range')
            relm.attrib['start'] = start
            relm.attrib['end'] = end
    return ET.ElementTree(root)
    

_XSLT = '''<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">
<xsl:output method="html"/>
<xsl:param name="modname"/>
<xsl:template match="/">
    <html><head><title>Coverage Results for <xsl:value-of select="$modname"/></title>
    <link rel="stylesheet" type="text/css" href="coverage.css"/></head>
    <body>
    <h2>Coverage Results for <xsl:value-of select="$modname"/></h2>
    <div class="annotated-source-code">
    <table width="100%">
    <xsl:for-each select="/coverageAnnotation/line">
        <xsl:variable name="lineno" select="position()"/>
        <xsl:choose>
            <xsl:when test="@executable='false'">
                <!--<div class="annotation-line non-exec">-->
                <tr class="annotation-line non-exec">
                <td class="non-exec"><xsl:value-of select="$lineno"/></td>
                <td><pre><xsl:value-of select="."/></pre></td></tr>
            </xsl:when>
            <xsl:when test="@executable='true' and @covered='true'">
                <tr class="annotation-line executable covered">
                <td class="covered"><xsl:value-of select="$lineno"/></td>
                <td><pre><xsl:value-of select="."/></pre></td></tr>
            </xsl:when>
            <xsl:otherwise>
                <tr class="annotation-line executable uncovered">
                <td class="uncovered"><xsl:value-of select="$lineno"/></td>
                <td><pre><xsl:value-of select="."/></pre></td></tr>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:for-each>
    </table>
    </div></body></html>
</xsl:template>
</xsl:stylesheet>'''


