# This file is part of xrayutilities.
#
# xrayutilities is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.
#
# Copyright (C) 2013 Dominik Kriegner <dominik.kriegner@gmail.com>

import os
import numpy

from . import __path__
from .material import Material,CubicElasticTensor,HexagonalElasticTensor,CubicAlloy
from . import lattice
from . import elements
from .cif import CIFFile
from .. import config

# some predefined materials
# PLEASE use N/m^2 as unit for cij for newly entered material ( 1 dyn/cm^2 = 0.1 N/m^2 = 0.1 GPa)
# Use Kelvin as unit for the Debye temperature
Si = Material("Si",lattice.DiamondLattice(elements.Si,5.43104),
                   CubicElasticTensor(165.77e+9,63.93e+9,79.62e+9),thetaDebye=640)
Ge = Material("Ge",lattice.DiamondLattice(elements.Ge,5.65785),
                   CubicElasticTensor(128.5e+9,48.3e+9,66.8e+9),thetaDebye=374)
InAs = Material("InAs",lattice.ZincBlendeLattice(elements.In,elements.As,6.0583),
                   CubicElasticTensor(8.34e+10,4.54e+10,3.95e+10),thetaDebye=280)
InP  = Material("InP",lattice.ZincBlendeLattice(elements.In,elements.P,5.8687),
                   CubicElasticTensor(10.11e+10,5.61e+10,4.56e+10),thetaDebye=425)
InSb  = Material("InSb",lattice.ZincBlendeLattice(elements.In,elements.Sb,6.47937),
                   CubicElasticTensor(6.66e+10,3.65e+10,3.02e+10),thetaDebye=160)
GaP  = Material("GaP",lattice.ZincBlendeLattice(elements.Ga,elements.P,5.4505),
                   CubicElasticTensor(14.05e+10,6.20e+10,7.03e+10),thetaDebye=445)
GaAs = Material("GaAs",lattice.ZincBlendeLattice(elements.Ga,elements.As,5.65325),
                   CubicElasticTensor(11.9e+10,5.34e+10,5.96e+10),thetaDebye=360)
GaSb = Material("GaSb",lattice.ZincBlendeLattice(elements.Ga,elements.Sb,6.09593),
                   CubicElasticTensor(8.83e+10,4.02e+10,4.32e+10),thetaDebye=266)
CdTe = Material("CdTe",lattice.ZincBlendeLattice(elements.Cd,elements.Te,6.482),
                   CubicElasticTensor(53.5,36.7,19.9)) # ? Unit of elastic constants
CdSe = Material("CdSe",lattice.WurtziteLattice(elements.Cd,elements.Se,4.300,7.011),
                   HexagonalElasticTensor(7.490e10,4.609e10,3.926e10,8.451e10,1.315e10))
CdSe_ZB = Material("CdSe ZB",lattice.ZincBlendeLattice(elements.Cd,elements.Se,6.052))
HgSe = Material("HgSe",lattice.ZincBlendeLattice(elements.Hg,elements.Se,6.085),
                   CubicElasticTensor(6.1e10,4.4e10,2.2e10))
PbTe = Material("PbTe",lattice.RockSalt_Cubic_Lattice(elements.Pb,elements.Te,6.464),
                   CubicElasticTensor(93.6,7.7,13.4))
PbSe = Material("PbSe",lattice.RockSalt_Cubic_Lattice(elements.Pb,elements.Se,6.128),
                   CubicElasticTensor(123.7,19.3,15.9))
GaN = Material("GaN",lattice.WurtziteLattice(elements.Ga,elements.N,3.189,5.186),
                   HexagonalElasticTensor(390.e9,145.e9,106.e9,398.e9,105.e9),thetaDebye=600)
BaF2 = Material("BaF2",lattice.CubicFm3mBaF2(elements.Ba,elements.F,6.2001))
MnTe = Material("MnTe",lattice.NiAsLattice(elements.Mn,elements.Te,4.1429,6.7031))
GeTe = Material("GeTe",lattice.GeTeRhombohedral(elements.Ge,elements.Te,5.996,88.18,0.237)) 
Al = Material("Al",lattice.FCCLattice(elements.Al,4.04958))
Au = Material("Au",lattice.FCCLattice(elements.Au,4.0782))
V = Material("V",lattice.BCCLattice(elements.V,3.024))
Ag2Se = Material("Ag2Se",lattice.NaumanniteLattice(elements.Ag,elements.Se,4.333,7.062,7.764))
VO2_Rutile = Material("VO_2",lattice.RutileLattice(elements.V,elements.O,4.55,2.88,0.305))
VO2_Baddeleyite = Material("VO_2",lattice.BaddeleyiteLattice(elements.V,elements.O,5.75,5.42,5.38,122.6))
SiO2 = Material("SiO_2",lattice.QuartzLattice(elements.Si,elements.O,4.916,4.916,5.4054))
In = Material("In",lattice.TetragonalIndiumLattice(elements.In,3.2523,4.9461))
Sb = Material("Sb",lattice.TrigonalR3mh(elements.Sb,4.307,11.273))
Sn = Material("Sn",lattice.TetragonalTinLattice(elements.Sn,5.8197,3.17488))
Ag = Material("Ag",lattice.FCCLattice(elements.Ag,4.0853))
SnAlpha = Material("Sn-alpha",lattice.DiamondLattice(elements.Sn,6.4912))
Cu = Material("Cu",lattice.FCCLattice(elements.Cu,3.61496))
CuMnAs = Material("CuMnAs",lattice.CuMnAsLattice(elements.Cu,elements.Mn,elements.As,3.82,3.82,6.30))
CaTiO3 = Material("CaTiO3",lattice.PerovskiteTypeRhombohedral(elements.Ca,elements.Ti,elements.O,3.795,90))
BiFeO3 = Material("BiFeO3",lattice.PerovskiteTypeRhombohedral(elements.Bi,elements.Fe,elements.O,3.965,89.3))

# materials defined from cif file
try:
    Al2O3 = Material("Al2O3", CIFFile(os.path.join(__path__[0],"data","Al2O3.cif")).Lattice())
except:
    if config.VERBOSITY >= config.INFO_LOW:
        print("XU.materials: Warning: import of CIF file based material failed")

# Alloys with special properties
class SiGe(CubicAlloy):
    def __init__(self,x):
        CubicAlloy.__init__(self,Si,Ge,x)

    def lattice_const_AB(self, latA, latB, x):
        """
        method to calculate the lattice parameter of the SiGe alloy with composition
        Si_{1-x}Ge_x
        """
        return latA+ (0.2*x+0.027*x**2)*latA/numpy.linalg.norm(latA)

    def _setxb(self,x):
        """
        method to set the composition of SiGe to Si_{1-x}Ge_x
        """
        if config.VERBOSITY >= config.DEBUG: print("XU.materials.SiGe._setxb: jump to base class")
        CubicAlloy._setxb(self,x)
        if config.VERBOSITY >= config.DEBUG: print("back from base class")
        #the lattice parameters need to be done in a different way
        a = self.lattice_const_AB(self.matA.lattice.a1[0], self.matB.lattice.a1[0], x)
        self.lattice = lattice.CubicLattice(a)
        self.rlattice = self.lattice.ReciprocalLattice()

    x = property(CubicAlloy._getxb,_setxb)
