from abc import ABCMeta, abstractmethod
import itertools


def _batches(it, n):
    """Generate batches (lists) of length n from iterable it."""
    it = iter(it)
    while True:
        batch = list(islice(it, n))
        if not batch:
            break
        yield batch


class DocumentSet(object):
    """Document set abstraction."""

    __metaclass__ = ABCMeta

    def batches(self, size=100):
        """Iterate over documents in batches."""
        return _batches(iter(self), size)


class DocumentList(object):
    """Thin wrapper around a list of strings."""

    def __init__(self, docs):
        self.docs = docs

    def __iter__(self):
        return iter(self.docs)


class QueryResults(DocumentSet):
    def __init__(self, es, query, index, type, field):
        query = {'query': query, 'fields': ['_id']}
        search = es[index][type]._search
        self.ids = [hit['_id']
                    for hit in search.get(data=query)['hits']['hits']
        self.es = es
        self.field = field
        self.index = index
        self.type = type

    def __iter__(self):
        for docid in self.ids:
            yield self.es[index][type][int(docid)].get()['_source'][field]
