import unittest
from textwrap import dedent

from src.add_remove import process_added_or_removed_element as sut
from test.mixins.assertion_mixin import PrintsAndResultsAssertionMixin


class AddUnitTests(PrintsAndResultsAssertionMixin, unittest.TestCase):

    def test_accepting_the_addition_of_a_single_top_level_key_value(self):
        yaml_map = {'thing1': 'Stuff'}
        elem_data = ('add', '', [('num_tokens', 256)])
        expected_result = {'thing1': 'Stuff', 'num_tokens': 256}
        expected_prints = dedent("""
        ***************
        Element(s) were added
         + num_tokens: 256
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_single_top_level_key_value(self):
        yaml_map = {'thing1': 'Stuff'}
        elem_data = ('add', '', [('num_tokens', 256)])
        expected_result = {'thing1': 'Stuff'}
        expected_prints = dedent("""
        ***************
        Element(s) were added
         + num_tokens: 256
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_a_single_nested_key_value(self):
        yaml_map = {'level1': {'level2': [{'thing1': 'The first thing'}]}}
        elem_data = ('add', 'level1.level2', [(1, {'thing2': 'The second thing'})])
        expected_result = {'level1': {'level2': [{'thing1': 'The first thing'},
                                                 {'thing2': 'The second thing'}]}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1.level2'
         + thing2: The second thing
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_single_nested_key_value(self):
        yaml_map = {'level1': {'level2': [{'thing1': 'The first thing'}]}}
        elem_data = ('add', 'level1.level2', [(1, {'thing2': 'The second thing'})])
        expected_result = {'level1': {'level2': [{'thing1': 'The first thing'}]}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1.level2'
         + thing2: The second thing
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_mutiple_nested_key_values(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': 'The stuff1'}}}
        elem_data = ('add', 'thing1.thing2', [('stuff2', 'The stuff2'), ('stuff3', 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': 'The stuff1',
                                                 'stuff2': 'The stuff2',
                                                 'stuff3': 'The stuff3'}}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1.thing2'
         + The stuff2
         + The stuff3
         """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_mutiple_nested_key_values(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': 'The stuff1'}}}
        elem_data = ('add', 'thing1.thing2', [('stuff2', 'The stuff2'), ('stuff3', 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': 'The stuff1'}}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1.thing2'
         + The stuff2
         + The stuff3
         """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_a_list(self):
        yaml_map = {'thing1': 'Stuff'}
        elem_data = ('add', '', [('data_file_directories', ['/usr/dirs1/', '/usr/dirs2/'])])
        expected_result = {'data_file_directories': ['/usr/dirs1/', '/usr/dirs2/'], 'thing1': 'Stuff'}
        expected_prints = dedent("""
        ***************
        Element(s) were added
         + data_file_directories: ['/usr/dirs1/', '/usr/dirs2/']
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_list(self):
        yaml_map = {'thing1': 'Stuff'}
        elem_data = ('add', '', [('data_file_directories', ['/usr/dirs1/', '/usr/dirs2/'])])
        expected_result = {'thing1': 'Stuff'}
        expected_prints = dedent("""
        ***************
        Element(s) were added
         + data_file_directories: ['/usr/dirs1/', '/usr/dirs2/']
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_a_nested_list(self):
        yaml_map = {'thing1': {'thing2': 'Some thing'}}
        elem_data = ('add', 'thing1', [('thing3', ['The stuff1', 'The stuff2'])])
        expected_result = {'thing1': {'thing2': 'Some thing', 'thing3': ['The stuff1', 'The stuff2']}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1'
         + The stuff1
         + The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_nested_list(self):
        yaml_map = {'thing1': {'thing2': 'Some thing'}}
        elem_data = ('add', 'thing1', [('thing3', ['The stuff1', 'The stuff2'])])
        expected_result = {'thing1': {'thing2': 'Some thing'}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1'
         + The stuff1
         + The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_a_top_level_item(self):
        yaml_map = {'level1': ['something/here']}
        elem_data = ('add', 'level1', [(1, 'something/new/goes/here')])
        expected_result = {'level1': ['something/here', 'something/new/goes/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1'
         + something/new/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_top_level_item(self):
        yaml_map = {'level1': ['something/here']}
        elem_data = ('add', 'level1', [(1, 'something/new/goes/here')])
        expected_result = {'level1': ['something/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1'
         + something/new/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_mutiple_top_level_items(self):
        yaml_map = {'thing1': ['The stuff1']}
        elem_data = ('add', 'thing1', [(1, 'The stuff2'), (2, 'The stuff3')])
        expected_result = {'thing1': ['The stuff1', 'The stuff2', 'The stuff3']}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1'
         + The stuff2
         + The stuff3
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_mutiple_top_level_items(self):
        yaml_map = {'thing1': ['The stuff1']}
        elem_data = ('add', 'thing1', [(1, 'The stuff2'), (2, 'The stuff3')])
        expected_result = {'thing1': ['The stuff1']}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1'
         + The stuff2
         + The stuff3
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_a_nested_item(self):
        yaml_map = {'level1': {'level2': ['something/here']}}
        elem_data = ('add', 'level1.level2', [(1, 'something/else/is/here')])
        expected_result = {'level1': {'level2': ['something/here', 'something/else/is/here']}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1.level2'
         + something/else/is/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_a_nested_item(self):
        yaml_map = {'level1': {'level2': ['something/here']}}
        elem_data = ('add', 'level1.level2', [(1, 'something/else/is/here')])
        expected_result = {'level1': {'level2': ['something/here']}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'level1.level2'
         + something/else/is/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_addition_of_mutiple_nested_items(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': ['The stuff1']}}}
        elem_data = ('add', 'thing1.thing2.stuff1', [(1, 'The stuff2'), (2, 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': ['The stuff1', 'The stuff2', 'The stuff3']}}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1.thing2.stuff1'
         + The stuff2
         + The stuff3
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_addition_of_mutiple_nested_items(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': ['The stuff1']}}}
        elem_data = ('add', 'thing1.thing2.stuff1', [(1, 'The stuff2'), (2, 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': ['The stuff1']}}}
        expected_prints = dedent("""
        ***************
        Element(s) were added to 'thing1.thing2.stuff1'
         + The stuff2
         + The stuff3
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_the_addition_of_mutiple_top_level_key_value_pairs_are_presented_to_the_user_individually(self):
        yaml_map = {'thing1': ['The stuff1']}
        elem_data = ('add', '', [('new_thing', 'hello'), ('rpc_server_type', 'sync')])
        expected_result = {'new_thing': 'hello', 'rpc_server_type': 'sync', 'thing1': ['The stuff1']}
        expected_prints = dedent("""
        ***************
        Element(s) were added
         + new_thing: hello


        ***************
        Element(s) were added
         + rpc_server_type: sync
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y", "y"])
