import unittest
from textwrap import dedent

from src.add_remove import process_added_or_removed_element as sut
from test.mixins.assertion_mixin import PrintsAndResultsAssertionMixin


class RemoveUnitTests(PrintsAndResultsAssertionMixin, unittest.TestCase):

    def test_accepting_the_removal_of_a_single_top_level_key_value(self):
        yaml_map = {'thing2': 'Other', 'thing1': 'Stuff'}
        elem_data = ('remove', '', [('thing2', 'Other')])
        expected_result = {'thing1': 'Stuff'}
        expected_prints = dedent("""
        ***************
        Element(s) were removed
         - thing2: Other
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_single_top_level_key_value(self):
        yaml_map = {'thing2': 'Other', 'thing1': 'Stuff'}
        elem_data = ('remove', '', [('thing2', 'Other')])
        expected_result = {'thing1': 'Stuff', 'thing2': 'Other'}
        expected_prints = dedent("""
        ***************
        Element(s) were removed
         - thing2: Other
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_a_single_nested_key_value(self):
        yaml_map = {'level1': {'level2': [{'thing1': 'The first thing'}, {'thing2': 'The second thing'}]}}
        elem_data = ('remove', 'level1.level2', [(1, {'thing2': 'The second thing'})])
        expected_result = {'level1': {'level2': [{'thing1': 'The first thing'}]}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1.level2'
         - thing2: The second thing
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_single_nested_key_value(self):
        yaml_map = {'level1': {'level2': [{'thing1': 'The first thing'}, {'thing2': 'The second thing'}]}}
        elem_data = ('remove', 'level1.level2', [(1, {'thing2': 'The second thing'})])
        expected_result = {'level1': {'level2': [{'thing1': 'The first thing'}, {'thing2': 'The second thing'}]}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1.level2'
         - thing2: The second thing
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_mutiple_nested_key_values(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': 'The stuff1',
                                          'stuff2': 'The stuff2',
                                          'stuff3': 'The stuff3'}}}
        elem_data = ('remove', 'thing1.thing2', [('stuff2', 'The stuff2'), ('stuff3', 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': 'The stuff1'}}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1.thing2'
         - The stuff2
         - The stuff3
         """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_mutiple_nested_key_values(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': 'The stuff1',
                                          'stuff2': 'The stuff2',
                                          'stuff3': 'The stuff3'}}}
        elem_data = ('remove', 'thing1.thing2', [('stuff2', 'The stuff2'), ('stuff3', 'The stuff3')])
        expected_result = {'thing1': {'thing2': {'stuff1': 'The stuff1',
                                                 'stuff2': 'The stuff2',
                                                 'stuff3': 'The stuff3'}}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1.thing2'
         - The stuff2
         - The stuff3
         """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_a_list(self):
        yaml_map = {'someList': ['one', 'two'], 'thing1': 'Stuff'}
        elem_data = ('remove', '', [('someList', ['one', 'two'])])
        expected_result = {'thing1': 'Stuff'}
        expected_prints = dedent("""
        ***************
        Element(s) were removed
         - someList: ['one', 'two']
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_list(self):
        yaml_map = {'someList': ['one', 'two'], 'thing1': 'Stuff'}
        elem_data = ('remove', '', [('someList', ['one', 'two'])])
        expected_result = {'thing1': 'Stuff', 'someList': ['one', 'two']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed
         - someList: ['one', 'two']
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_a_nested_list(self):
        yaml_map = {'thing1': {'thing2': 'Some thing', 'thing3': ['The stuff1', 'The stuff2']}}
        elem_data = ('remove', 'thing1', [('thing3', ['The stuff1', 'The stuff2'])])
        expected_result = {'thing1': {'thing2': 'Some thing'}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1'
         - The stuff1
         - The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_nested_list(self):
        yaml_map = {'thing1': {'thing2': 'Some thing', 'thing3': ['The stuff1', 'The stuff2']}}
        elem_data = ('remove', 'thing1', [('thing3', ['The stuff1', 'The stuff2'])])
        expected_result = {'thing1': {'thing2': 'Some thing', 'thing3': ['The stuff1', 'The stuff2']}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1'
         - The stuff1
         - The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_a_top_level_item(self):
        yaml_map = {'level1': ['something/here', 'something/else/goes/here']}
        elem_data = ('remove', 'level1', [(1, 'something/else/goes/here')])
        expected_result = {'level1': ['something/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1'
         - something/else/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_top_level_item(self):
        yaml_map = {'level1': ['something/here', 'something/else/goes/here']}
        elem_data = ('remove', 'level1', [(1, 'something/else/goes/here')])
        expected_result = {'level1': ['something/here', 'something/else/goes/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1'
         - something/else/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_remove_of_mutiple_top_level_items(self):
        yaml_map = {'level1': ['thing/here', 'thing/else/goes/here', 'thing/more/goes/here']}
        elem_data = ('remove', 'level1', [(2, 'thing/more/goes/here'), (1, 'thing/else/goes/here')])
        expected_result = {'level1': ['thing/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1'
         - thing/more/goes/here
         - thing/else/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_remove_of_mutiple_top_level_items(self):
        yaml_map = {'level1': ['thing/here', 'thing/more/goes/here', 'thing/else/goes/here']}
        elem_data = ('remove', 'level1', [(2, 'thing/more/goes/here'), (1, 'thing/else/goes/here')])
        expected_result = {'level1': ['thing/here', 'thing/more/goes/here', 'thing/else/goes/here']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1'
         - thing/more/goes/here
         - thing/else/goes/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_a_nested_item(self):
        yaml_map = {'level1': {'level2': ['something/here', 'something/else/here']}}
        elem_data = ('remove', 'level1.level2', [(1, 'something/else/here')])
        expected_result = {'level1': {'level2': ['something/here']}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1.level2'
         - something/else/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_a_nested_item(self):
        yaml_map = {'level1': {'level2': ['something/here', 'something/else/here']}}
        elem_data = ('remove', 'level1.level2', [(1, 'something/else/here')])
        expected_result = {'level1': {'level2': ['something/here', 'something/else/here']}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'level1.level2'
         - something/else/here
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_accepting_the_removal_of_mutiple_nested_items(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': ['The stuff1', 'The stuff2', 'The stuff3']}}}
        elem_data = ('remove', 'thing1.thing2.stuff1', [(2, 'The stuff3'), (1, 'The stuff2')])
        expected_result = {'thing1': {'thing2': {'stuff1': ['The stuff1']}}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1.thing2.stuff1'
         - The stuff3
         - The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y"])

    def test_decline_the_removal_of_mutiple_nested_items(self):
        yaml_map = {'thing1': {'thing2': {'stuff1': ['The stuff1', 'The stuff2', 'The stuff3']}}}
        elem_data = ('remove', 'thing1.thing2.stuff1', [(2, 'The stuff3'), (1, 'The stuff2')])
        expected_result = {'thing1': {'thing2': {'stuff1': ['The stuff1', 'The stuff2', 'The stuff3']}}}
        expected_prints = dedent("""
        ***************
        Element(s) were removed from 'thing1.thing2.stuff1'
         - The stuff3
         - The stuff2
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["n"])

    def test_the_removal_of_mutiple_top_level_key_value_pairs_are_presented_to_the_user_individually(self):
        yaml_map = {'thing1': ['The stuff1'], 'new_thing': ['hello'], 'rpc_server_type': ['sync']}
        elem_data = ('remove', '', [('new_thing', 'hello'), ('rpc_server_type', 'sync')])
        expected_result = {'thing1': ['The stuff1']}
        expected_prints = dedent("""
        ***************
        Element(s) were removed
         - new_thing: hello


        ***************
        Element(s) were removed
         - rpc_server_type: sync
        """)[1:-1]
        self.validate(sut, yaml_map, elem_data, expected_result, expected_prints, ["y", "y"])
