"""Utility classes that do not fit elsewhere"""

__author__  = "Tamas Nepusz"
__email__   = "tamas@cs.rhul.ac.uk"
__copyright__ = "Copyright (c) 2010, Tamas Nepusz"
__license__ = "MIT"

from collections import deque
from functools import wraps

def axis_label(label):
    """Creates a decorator that attaches an attribute named ``__axis_label__``
    to a function. This is used later in the plotting functions to derive an
    appropriate axis label if the function is plotted on an axis.

    Usage::

        @axis_label("x squared")
        def f(x):
            return x ** 2
    """
    def result(func):
        func.__axis_label__ = label
        return func
    return result


def endless_generator(func, *args, **kwds):
    """Constructs an endless generator that yields elements generated by
    `func`. At each invocation, `func` should return a list of items
    that will be yielded one by one by this generator. Whenever the
    generator runs out of items, it will call `func` again for a new
    set of items. Additional positional and keyword arguments are passed
    on intact to `func`.
    """
    buffer = deque()
    while True:
        while buffer:
            yield buffer.popleft()
        buffer.extend(func(*args, **kwds))


def vectorized(func):
    """Decorator that returns a vectorized variant of a single-argument
    function.
    """
    @wraps(func)
    def wrapper(item, *args, **kwds):
        if hasattr(item, "__iter__"):
            return [func(i) for i in item]
        return func(item)
    return wrapper

