from xml.sax.handler import ContentHandler as _ContentHandler
from cStringIO import StringIO as _StringIO

class Element(object):
	def __init__(self, qname, attributes=None, namespaces={}, parent=None):
		self.parent = parent
		self.qname = qname
		self.text = ''		
		self.children = []		
		self.attributes = {}
		
		self.namespaces = {}
		self.namespaces.update(namespaces)
		
		if attributes:
			_attributes = [x for x in attributes.items() if not x[0].startswith('xmlns:')]
			_namespaces = [x for x in attributes.items() if x[0].startswith('xmlns:')]			
			
			for aname, avalue in _namespaces:
				self[aname] = avalue
				
			for aname, avalue in _attributes:
				self[aname] = avalue
		
	def map(self, prefix, uri):
		"""Binds the supplied namespace prefix to the supplied URI.
		
		Any namespaces used in an element or its sub-elements must
		be bound before use.
		"""
		
		self.namespaces[prefix] = uri
	
	def __getitem__(self, name):
		if ':' in name:
			ns, localname = name.split(':')
		else:
			ns = ''
			localname = name
			
		if name.startswith('xmlns'):
			# lookup namespace mapping
			_ns = self.find_ns(localname)
			
			if not _ns:
				raise UndeclaredNamespaceException(ns)
			else:
				return _ns[1]
			
		return self.attributes[name]
	
	def __setitem__(self, name, value):
		if ':' in name:
			ns, localname = name.split(':')
			
			if not self.find_ns(ns) and ns != 'xmlns':
				raise UndeclaredNamespaceException('%s was not declared prior to use with localname %s' % (ns, localname))
			elif ns == 'xmlns':
				self.namespaces[localname] = value
				return
		
		self.attributes[name] = str(value)		
	
	def __delitem__(self, name):
		if name.startswith('xmlns:'):
			k = name[len('xmlns:'):]			
			if len([x for x in self.attributes.items() if x[0].startswith('%s:' % k)]) > 0:			
				raise NamespaceNotDeletableException('Could not delete namespace %s - some elements still belong to it' % name)
		
		if name.startswith('xmlns:'):
			ns, localname = name.split(':')
			
			if self.namespaces.has_key(localname):
				del self.namespaces[localname]			
		else:
			del self.attributes[name]
	
	def find_ns(self, prefix):
		"""Returns the URI bound to the supplied prefix or None if there is
		no binding for the prefix.
		
		If this node has the supplied prefix mapped its value is returned;
		otherwise, this node's ancestors are searched until the root of the
		tree is reached.		
		"""
		
		if self.namespaces.has_key(prefix):
			return self.namespaces[prefix]
		elif self.parent and not self.namespaces.has_key(prefix):
			return self.parent.find_ns(prefix)
		else:
			return None
	
	def lookup_prefix(self, uri):
		"""Returns the prefix bound to the supplied URI or None if there is
		no binding for the URI.
		
		If this node has a prefix bound to the supplied URI it is returned;
		otherwise, this node's ancestors are searched until the root of the
		tree is reached.
		"""
		
		for _p, _u in self.namespaces.items():
			if _u == uri:
				return _p
				
		if self.parent:
			return self.parent.lookup_prefix(uri)
		else:
			return None
	
	def append(self, qname, attributes=None, text=None):
		"""Append a child node to this node
		
		Raises an exception if the qname contains an un-specified
		XML namespace.
		"""		
		if ':' in qname:
			xmlns, localname = qname.split(':')
			
			if xmlns not in self.namespaces.keys():
				raise UndeclaredNamespaceException('%s was not declared prior to use' % xmlns)
		
		elem = Element(qname, attributes, parent=self)
		self.children.append(elem)
		
		if text:
			elem.text = text
		
		return elem
	
	def write(self, f):		
		f.write('<%s' % self.qname)

		for namespace in self.namespaces.items():
			f.write(' xmlns:%s="%s"' % namespace)
			
		for attribute in self.attributes.items():
			f.write(' %s="%s"' % attribute)

		if len(self.children) == 0 and not self.text:
			f.write(' />')		
		else:
			f.write('>')
			
			if self.text:
				f.write(self.text)

			for child in self.children:				
				child.write(f)

			f.write('</%s>' % self.qname)
	
	def __repr__(self, asdoc=False):
		f = _StringIO()
		self.write(f)
		xmlstring = f.getvalue().strip()
		r = '%s'
		
		if asdoc:
			r = '<?xml version="1.0"?>%s'
			
		return r % xmlstring			
	
	def asdoc(self):
		return self.__repr__(True)
	
	def __eq__(self, elem):		
		if elem.qname != self.qname:
			#print 'QNames did not match (%s != %s)' % (self.qname, elem.qname)
			return False
		elif elem.attributes != self.attributes:
			#print 'Attributes did not match (%s != %s)' % (self.attributes, elem.attributes)
			return False
		elif elem.children != self.children:
			#print 'Children did not match (%s != %s)' % (self.children, elem.children)
			return False
		else:
			return True
		
class UndeclaredNamespaceException(Exception):
	def __init__(self, prefix):
		Exception.__init__(self, '%s was not bound to a URI before use' % prefix)

class NamespaceNotDeletableException(Exception):
	pass

class __XMLBuilder(_ContentHandler):
	def __init__(self):		
		self.tree = None
		self.elem = None
		self.namespaces = {}
	
	def startPrefixMapping(self, prefix, uri):
		self.namespaces[prefix] = uri
		#print 'Started mapping %s to %s' % (prefix, uri)
		
	def endPrefixMapping(self, prefix):
		del self.namespaces[prefix]
	
	def startElementNS(self, name, qname, attrs):
		elem = Element(qname, namespaces=self.namespaces)
		
		for k,v in attrs.items():			
			ns, localname = k			
			
			if ns:
				p = elem.lookup_prefix(ns)
				elem['%s:%s' % (p, localname)] = v
			else:
				elem[localname] = v
		
		if not self.tree:
			self.tree = elem
			self.elem = elem
		else:
			self.elem.children.append(elem)

def parse(xml):
	from xml.sax.handler import feature_namespaces
	from xml.sax import make_parser
	
	builder = __XMLBuilder()
	
	parser = make_parser()
	parser.setFeature(feature_namespaces, True)
	parser.setContentHandler(builder)
	parser.setErrorHandler(builder)
	
	f = _StringIO(xml)
	parser.parse(f)
	
	return builder.tree