#!/usr/bin/env python

import argparse
from pay import Yollapay
from response import YollapayResponse
import os
import sys
import json
import command

__version__ = VERSION = version = '0.0.2'


def main():
    parser = argparse.ArgumentParser(prog='yollapay',
                                     add_help=True,
                                     formatter_class=argparse.RawDescriptionHelpFormatter,
                                     epilog="""Learn more at \033[92mhttp://dev.yollapay.com\033[0m""")
    parser.add_argument('-v', '--verbose', help='verbose', action='store_true', default=False)
    parser.add_argument('--version', action='version', version='%(prog)s '+version)
    parser.add_argument('--json',
                        action='store_true',
                        default=False,
                        help="express output as json (more data is included)")
    parser.add_argument('-a', '--account',
                        default=os.getenv('YOLLAPAY_ACCOUNT'),
                        help="on the behalf of this account")

    subparsers = parser.add_subparsers(title='Yollapay Commands')

    # ---------------
    # Account
    # ---------------

    # account
    parser_account = subparsers.add_parser('account',
                                           help='return a list of all accounts')
    parser_account.set_defaults(func=command.account)
    parser_account.add_argument('name',
                                nargs="?",
                                help="show information about single account")

    # account:add
    parser_account_add = subparsers.add_parser('account:add',
                                               help='add a new client account')
    parser_account_add.set_defaults(func=command.account_add)
    parser_account_add.add_argument('name',
                                    nargs=1,
                                    help="account id")
    # account:update
    parser_account_update = subparsers.add_parser('account:update',
                                                  help='update customer information')
    parser_account_update.set_defaults(func=command.account_update)
    parser_account_update.add_argument('name',
                                       nargs=1,
                                       help="account id")

    # ---------------
    # Batch
    # ---------------

    # batch
    parser_batch = subparsers.add_parser('batch',
                                         help='view current batch details')
    parser_batch.set_defaults(func=command.batch)
    parser_batch.add_argument('args',
                              nargs='*',
                              help="batch single transactions")
    parser_batch.add_argument('-a', "--all",
                              action='store_true',
                              dest="all",
                              help="return list of all transactions in batch")

    # batch:all
    parser_batch = subparsers.add_parser('batch:all',
                                         help='submit all preauth transactions to batch')
    parser_batch.set_defaults(func=command.batch_all)


    # ---------------
    # Config
    # ---------------
    parser_config = subparsers.add_parser('config',
                                           help='configure your Yollapay API account')
    parser_config.set_defaults(func=command.config)
    # ---------------
    # Credit
    # ---------------

    # preauth
    parser_preauth = subparsers.add_parser('credit:preauth',
                                           help='run new transaction as preauth')
    parser_preauth.set_defaults(func=command.credit_preauth)
    parser_preauth.add_argument('--json',
                                action='store_true',
                                default=False,
                                help="express output as json (more data is included)")
    parser_preauth_group = parser_preauth.add_mutually_exclusive_group(required=True)
    parser_preauth_group.add_argument('-n', "--number",
                                      dest="number",
                                      metavar="",
                                      help="credit card number")
    parser_preauth.add_argument('-e', "--expires",
                                dest="expires",
                                metavar="mmyyyy",
                                help="credit card expire date, accepts all types of formats")
    parser_preauth.add_argument('-a', "--amount",
                                dest="amount",
                                metavar="xx.xx",
                                required=True,
                                type=float,
                                help="charge amount")
    parser_preauth_group.add_argument('-s', "--swipe",
                                dest="swipe",
                                metavar="",
                                help="raw swipe data")

    parser_purchase = subparsers.add_parser('credit:purchase', help='run new transaction as purchase')
    parser_void = subparsers.add_parser('credit:void', help='run new transaction as void')
    parser_refund = subparsers.add_parser('credit:refund', help='run new transaction as refund')
    parser_complete = subparsers.add_parser('credit:complete', help='complete a previous preauth')

    # ---------------
    # Gateway
    # ---------------

    # gateway
    parser_gateway = subparsers.add_parser('gateway',
                                           help='show installed gateways')
    parser_gateway.set_defaults(func=command.gateway)

    # gateway:add
    parser_gateway_add = subparsers.add_parser('gateway:add',
                                              help="add a new gateway")
    parser_gateway_add.set_defaults(func=command.gateway_add)
    parser_gateway_add.add_argument('gateway',
                                    nargs='?',
                                    help="the gateway to be added and configured")
    parser_gateway_remove = subparsers.add_parser('gateway:remove',
                                                  help="delete a gateway")
    parser_gateway_remove.add_argument('gateway',
                                       nargs=1,
                                       help="gateway to remove")
    parser_gateway_update = subparsers.add_parser('gateway:update',
                                                  help="update gateway information")
    parser_gateway_update.add_argument('gateway',
                                       nargs=1,
                                       help="gateway to update configuration")
    
    # ---------------
    # Keys
    # ---------------
    parser_keys = subparsers.add_parser('keys', help='return a list of access keys')
    parser_keys_add = subparsers.add_parser('keys:add', help='add a new access key')
    parser_keys_remove = subparsers.add_parser('keys:remove', help='remove an access key')

    # ---------------
    # Search
    # ---------------

    # Search
    parser_search = subparsers.add_parser('search',
                                          help='search for transactions')
    parser_search.set_defaults(func=command.search)
    parser_search.add_argument('transactions',
                               nargs='*',
                               help="list of transactions to batch")
    parser_search.add_argument('-t', "--time",
                               dest="time",
                               help="search a specfic time range")

    if not os.getenv("YOLLAPAY_API_KEY"):
        # process config file
        config_file = os.path.join(os.getenv('HOME'), ".yollapay")
        if os.path.exists(config_file):
            c = json.load(config_file)
            os.putenv("YOLLAPAY_API_KEY", c['api_key'])
            os.putenv("YOLLAPAY_API_SECRET", c['api_secret'])
        else:
            """config file required
            go through welcome"""
            pass

    if len(sys.argv) == 1:
        parser.print_help()
    else:
        args = parser.parse_args()
        args.func(args)

if __name__ == "__main__":
    main()
