import os
import requests
import json
from response import YollapayResponse
import valideer
import datetime
import timestring
import decimal

TornadoAsyncHTTPClient = None
TornadoHTTPResponse = None
try:
    from tornado.httpclient import AsyncHTTPClient as TornadoAsyncHTTPClient
    from tornado.httpclient import HTTPResponse as TornadoHTTPResponse
except ImportError:
    TornadoAsyncHTTPClient = None
    TornadoHTTPResponse = None


class Money(valideer.String):
    name = "money"
    def __init__(self):
        super(Money, self).__init__()
    
    def validate(self, value, adapt=True):
        try:
            float(value)
        except ValueError:
            raise valideer.ValidationError("Value must be a valid number")
        if float(value) < 0:
            raise valideer.ValidationError("Value must be >= 0")
        return "%.2f" % float(value)


class Yollapay:
    LIVE_ENDPOINT = "https://pay.yollapay.com/api/1"
    DEV_ENDPOINT = "https://demo.yollapay.com/api/1"
    _DATA_VALIDATORS = dict(purchase=valideer.parse({"+amount": "money",
                                                     "?expires": valideer.Pattern(r"[01]\d{3}"),
                                                     "?name": valideer.String(min_length=1, max_length=30),
                                                     "?number": valideer.Pattern(r"^\d{15,16}$"),
                                                     "?cvv": valideer.Pattern(r"^\d{3,4}$"),
                                                     "?swipe": valideer.Pattern(r"^%B\d{0,19}\^[\w\s\/]{2,26}\^\d{7}\w*\?\n;\d{0,19}=\d{7}\w*\?"),
                                                     "+type": "string"}, additional_properties=False),
                            preauth=valideer.parse({"+amount": "money",
                                                     "?expires": valideer.Pattern(r"[01]\d{3}"),
                                                     "?name": valideer.String(min_length=1, max_length=30),
                                                     "?number": valideer.Pattern(r"^\d{15,16}$"),
                                                     "?cvv": valideer.Pattern(r"^\d{3,4}$"),
                                                     "?swipe": valideer.Pattern(r"^%B\d{0,19}\^[\w\s\/]{2,26}\^\d{7}\w*\?\n;\d{0,19}=\d{7}\w*\?"),
                                                     "+type": "string"}, additional_properties=False),
                            complete=valideer.parse({"+id": valideer.Pattern(r"^\w{8}-\w{4}-\w{4}-\w{4}-\w{12}$"),
                                                     "?tip": "money"}, additional_properties=False), 
                            void=valideer.parse({"+id": valideer.Pattern(r"^\w{8}-\w{4}-\w{4}-\w{4}-\w{12}$"),
                                                 "+type": "string"}), additional_properties=False)

    def __init__(self, api_key=None, api_secret=None, **kwargs):
        """Initialize the Yollapay by passing in your api key and api secret.
        """
        if api_key is None:
            api_key = os.getenv("YOLLAPAY_API_KEY")
            assert api_key, "No yollapay api key was found."
        self._api_key = api_key

        if api_secret is None:
              api_secret = os.getenv("YOLLAPAY_API_SECRET")
              assert api_secret, "No yollapay api secret was found."
        self._api_secret = api_secret

        self._timeout = kwargs.get('timeout', 20)
        self._test = kwargs.get('test',
                                os.getenv('YOLLAPAY_API_TEST') == 'TRUE')

    # ------------------------------
    # Processing Methods
    # ------------------------------
    def credit(self, data, account=None, httpclient=None, callback=None):
        """Process a credit card.

        ex. 
        ```
        yollapay.Yollapay().credit(type="preauth",
                                   number="4111111111111111",
                                   expires="07/15",
                                   amount=10)
        ```
        """
        # Fix expires value
        if data.get('expires'):
            try:
                # dates from credit card show up in this format
                data['expires'] = datetime.datetime.strptime(data['expires'], "%y%m")
            except ValueError:
                try:
                    data['expires'] = timestring.Date(data['expires']).date
                except ValueError:
                    error = valideer.ValidationError("Failed to parse expiration date")
                    error.add_context('expires')
                    raise error
            data['expires'] = data['expires'].strftime("%m%y")
        # /expires

        # Pre validation
        self._DATA_VALIDATORS[data['type']].validate(data)

        # Send api request
        return self._api_request('/credit',
                                 data=data,
                                 account=account,
                                 httpclient=httpclient,
                                 callback=callback)

    def _api_request(self, path, data, account, method="POST", httpclient=None, callback=None):
        url = self.endpoint + path
        assert method in ("POST", "GET")
        assert type(data) is dict
        headers = {"Yollapay-Token": "%s:%s" % (self._api_key, self._api_secret),
                   "Yollapay-Account": account or os.getenv('YOLLAPAY_ACCOUNT'),
                   "Accept": "application/json",
                   "Content-Type": "application/json"}
        assert headers['Yollapay-Account'] is not None, "Acccount must be a valid Yollapay Account ID"
        
        # warp me in a try
        body = json.dumps(data)

        self.callback = callback

        if httpclient:
            if TornadoAsyncHTTPClient is not None and isinstance(httpclient, TornadoAsyncHTTPClient):
                # a tornado httpclient was found
                httpclient.fetch(url,
                                 body=body,
                                 headers=headers,
                                 method=method,
                                 request_timeout=self.timeout,
                                 callback=self._parse_response)
            else:
                raise ValueError("Provided httpclient is not supported.")
        else:
            if method == "POST":
                resp = requests.post(url, headers=headers, data=body, timeout=self.timeout)
            else:
                resp = requests.get(url,  headers=headers, data=body, timeout=self.timeout)
            return self._parse_response(resp)

    def _parse_response(self, response):
        """Parse the body from the api requests
        """
        if TornadoHTTPResponse and isinstance(response, TornadoHTTPResponse):
            # extract the response body
            response = YollapayResponse(json.loads(response.body))
        else:
            response = YollapayResponse(response.json())
        # send data back to callback, if callable
        if hasattr(self.callback, "__call__"):
            self.callback(response)
        else:
            return response

    # ------------------------------
    # Properties
    # ------------------------------
    @property
    def endpoint(self):
        return os.getenv("YOLLAPAY_ENDPOINT",
                         self.LIVE_ENDPOINT if self._test \
                         else self.DEV_ENDPOINT)

    @property
    def timeout(self):
        return self._timeout

    @timeout.setter
    def timeout(self, timeout):
        if type(timeout) is not int:
            raise ValueError("Invalid value for timeout. Must be type int.")
        self._timeout = float(timeout)

    @property
    def callback(self):
        return self._callback

    @callback.setter
    def callback(self, callback):
        if callback is None:
            self._callback = None
        if not hasattr(callback, "__call__"):
            raise ValueError("Invalid value for callback. Must be a callable method.")
        self._callback = callback

