import os
import requests
import json
from response import YollapayResponse
import valideer
import datetime
import timestring

TornadoAsyncHTTPClient = None
TornadoHTTPResponse = None
try:
    from tornado.httpclient import AsyncHTTPClient as TornadoAsyncHTTPClient
    from tornado.httpclient import HTTPResponse as TornadoHTTPResponse
except ImportError:
    TornadoAsyncHTTPClient = None
    TornadoHTTPResponse = None


class Money(valideer.String):
    name = "money"
    def __init__(self):
        super(Money, self).__init__()
    
    def validate(self, value, adapt=True):
        try:
            float(value)
        except ValueError:
            raise valideer.ValidationError("Value must be a valid number")
        if float(value) < 0:
            raise valideer.ValidationError("Value must be >= 0")
        return "%.2f" % float(value)


API_KEY = os.getenv("YOLLAPAY_API_KEY")
API_SECRET = os.getenv("YOLLAPAY_API_SECRET")
ENDPOINT = os.getenv("YOLLAPAY_ENDPOINT", "https://demo.yollapay.com")

def init(api_key, api_secret, endpoint=None):
    global API_KEY
    global API_SECRET
    global ENDPOINT
    
    API_KEY = api_key
    API_SECRET = api_secret

    if endpoint:
        ENDPOINT = endpoint
    elif os.getenv("YOLLAPAY_ENDPOINT"):
        ENDPOINT = os.getenv("YOLLAPAY_ENDPOINT")

    assert ENDPOINT is not None, "No yollapay endpoint set"
    assert ENDPOINT.startswith('http'), "Invalid yollapay api endpoint"
    if ENDPOINT.endswith('/'):
        ENDPOINT = ENDPOINT[:-1]

def yoddle(_f):
    """API wrapper for consistancy for all api calls
    """
    @classmethod
    def decore(self, *args, **kwargs):
        """Removes common arguments for all api calls
        """
        account = kwargs.get('account')
        httpclient = kwargs.get('httpclient')
        callback = kwargs.get('callback')
        if 'account' in kwargs:
            del kwargs['account']
        if 'httpclient' in kwargs:
            del kwargs['httpclient']
        if 'callback' in kwargs:
            del kwargs['callback']

        # construct Yollapay
        _c = self(account, httpclient, callback)

        # get the api data for the api request
        data, method = _f(_c, *args, **kwargs)

        # run the api request, return the results
        # using async will return None
        return _c._api_request(path="/%s/%s" % (self.__name__, _f.__name__),
                               data=data,
                               method=method.upper())

    return decore

class YollapayAPI(object):
    """Base Yollapay object that that is used by every api call.
    """
    def __init__(self, account, httpclient, callback):
        global API_KEY
        global API_SECRET
        self._api_key = API_KEY or os.getenv("YOLLAPAY_API_KEY")
        assert self._api_key, "No yollapay api key was found."

        self._api_secret = API_SECRET or os.getenv("YOLLAPAY_API_SECRET")
        assert self._api_secret, "No yollapay api secret was found."

        self.account = account or os.getenv('YOLLAPAY_ACCOUNT')
        self.httpclient = httpclient
        self.callback = callback

    def _api_request(self, path, data, method="POST"):
        url = self.endpoint + path
        assert method in ("POST", "GET")
        assert type(data) is dict
        headers = {"Yollapay-Token": "%s:%s" % (self._api_key, self._api_secret),
                   "Yollapay-Account": self.account,
                   "Accept": "application/json",
                   "Content-Type": "application/json"}
        assert headers['Yollapay-Account'] is not None, "Acccount must be a valid Yollapay Account ID"
        
        # warp me in a try
        body = json.dumps(data)

        if self.httpclient:
            if TornadoAsyncHTTPClient is not None and isinstance(self.httpclient, TornadoAsyncHTTPClient):
                # a tornado httpclient was found
                self.httpclient.fetch(url,
                                      body=body,
                                      headers=headers,
                                      method=method,
                                      request_timeout=45,
                                      callback=self._parse_response)
            else:
                raise ValueError("Provided httpclient is not supported.")
        else:
            if method == "POST":
                resp = requests.post(url, headers=headers, data=body, timeout=45)
            else:
                resp = requests.get(url,  headers=headers, data=body, timeout=45)
            return self._parse_response(resp)

    def _parse_response(self, response):
        """Parse the body from the api requests
        """
        if TornadoHTTPResponse and isinstance(response, TornadoHTTPResponse):
            # extract the response body
            response = YollapayResponse(json.loads(response.body))
        else:
            response = YollapayResponse(response.json())
        # send data back to callback, if callable
        if hasattr(self.callback, "__call__"):
            self.callback(response)
        else:
            return response

    # ------------------------------
    # Properties
    # ------------------------------
    @property
    def endpoint(self):
        global ENDPOINT
        return ENDPOINT

    @property
    def callback(self):
        return self._callback

    @callback.setter
    def callback(self, callback):
        if callback is None:
            self._callback = None
        elif not hasattr(callback, "__call__"):
            raise ValueError("Invalid value for callback. Must be a callable method.")
        else:
            self._callback = callback

class credit(YollapayAPI):
    """All credit api calls. 
    /credit/:preauth|complete|sale
    """
    _DATA_VALIDATORS = dict(purchase=valideer.parse({"+amount": "money",
                                                     "?expires": valideer.Pattern(r"[01]\d{3}"),
                                                     "?name": valideer.String(min_length=1, max_length=30),
                                                     "?number": valideer.Pattern(r"^\d{15,16}$"),
                                                     "?cv": valideer.Pattern(r"^\d{3,4}$"),
                                                     "?zip": valideer.Pattern(r"^\d{5}$"),
                                                     "?swipe": valideer.Pattern(r"^%B\d{0,19}\^[\w\s\/]{2,26}\^\d{7}\w*\?\n;\d{0,19}=\d{7}\w*\?"),
                                                     "+type": "string"}, additional_properties=False),
                            preauth=valideer.parse({"+amount": "money",
                                                     "?expires": valideer.Pattern(r"[01]\d{3}"),
                                                     "?name": valideer.String(min_length=1, max_length=30),
                                                     "?number": valideer.Pattern(r"^\d{15,16}$"),
                                                     "?zip": valideer.Pattern(r"^\d{5}$"),
                                                     "?cv": valideer.Pattern(r"^\d{3,4}$"),
                                                     "?swipe": valideer.Pattern(r"^%B\d{0,19}\^[\w\s\/]{2,26}\^\d{7}\w*\?\n;\d{0,19}=\d{7}\w*\?"),
                                                     "+type": "string"}, additional_properties=False),
                            complete=valideer.parse({"+id": valideer.Pattern(r"^\w{8}-\w{4}-\w{4}-\w{4}-\w{12}$"),
                                                     "?tip": "money"}, additional_properties=False), 
                            void=valideer.parse({"+id": valideer.Pattern(r"^\w{8}-\w{4}-\w{4}-\w{4}-\w{12}$"),
                                                 "+type": "string"}), additional_properties=False)


    # ------------------------------
    # Processing Methods
    # ------------------------------
    @yoddle
    def preauth(self, **data):
        """Process a credit card for preauth
        yollapay.credit.preauth(number="4111111111111111",
                                expires="07/15",
                                amount=10)

        yollapay.credit.preauth(swipe="<raw swipe data>",
                                amount=10)
        """
        data['type'] = 'preauth'
        
        # Fix expires value
        if data.get('expires'):
            try:
                # dates from credit card show up in this format
                data['expires'] = datetime.datetime.strptime(data['expires'], "%y%m")
            except ValueError:
                try:
                    data['expires'] = timestring.Date(data['expires']).date
                except ValueError:
                    error = valideer.ValidationError("Failed to parse expiration date")
                    error.add_context('expires')
                    raise error
            data['expires'] = data['expires'].strftime("%m%y")
        # /expires

        # Pre validation
        self._DATA_VALIDATORS[data['type']].validate(data)

        # Send api request
        return data, 'post'

class account(YollapayAPI):
    """All account api methods
    """
    @yoddle
    def add(self, _id, name):
        return dict(id=_id, name=name), 'POST'

    @yoddle
    def remove(self, _id):
        return dict(id=_id), 'POST'

    @yoddle
    def get(self, _id=None):
        return dict(id=_id), 'GET'

    @yoddle
    def set(self, field=None, value=None):
        return dict(field=field, value=value), 'post'

class gateway(YollapayAPI):
    @yoddle
    def add(self, gateway, username, password, key=None, secret=None):
        return dict(gateway=gateway, username=username, password=password, key=key, secret=secret), 'post'
