# -*- coding: UTF-8 -*-
from zabbix_client import ZabbixServerProxy
from powerline.theme import requires_segment_info

CRITICAL = 'critical'
WARNINGS = 'warnings'

CRITICAL_SYMBOLE = u'🔥'
WARNINGS_SYMBOLE = u'⚠️'

SYMBOLE_NODIFF = u'±'
SYMBOLE_ADD = u'+'
SYMBOLE_NEGATIVE = u'-'


class ZabbixAgent(object):

    def __init__(self, username, password, endpoint):
        self.conn = ZabbixServerProxy(endpoint)
        self.conn.user.login(user=username, password=password)

    def get_current_high_triggers(self):

        return self.conn.trigger.get(
            filter={'priority': [4, 5]},
            only_true='true', monitored=1, withUnacknowledgedEvents='true')

    def get_current_warning_triggers(self):

        return self.conn.trigger.get(
            filter={'priority': [1, 2, 3]},
            only_true='true', monitored=1, withUnacknowledgedEvents='true')


def zabbix_current_active(pl, username, password, endpoint):

    zabbix = ZabbixAgent(username, password, endpoint)
    triggers_high = zabbix.get_current_high_triggers()
    triggers_low = zabbix.get_current_warning_triggers()

    if triggers_high:
        level = 100
    elif triggers_low:
        level = triggers_low
    else:
        level = 0

    return [{
        'contents': u"H[%s] W[%s]" % (len(triggers_high), len(triggers_low)),
        'highlight_group': [
            'zabbix_current_state_gradient', 'zabbix_current_state'],
        'divider_highlight_group': 'background:divider',
        'gradient_level': level
    }]

@requires_segment_info
def active_triggers(
    pl, segment_info, username, password, endpoint, triggers='warnings'):
    zabbix = ZabbixAgent(username, password, endpoint)

    if triggers == WARNINGS:
        triggers = zabbix.get_current_warning_triggers()
        symbole = WARNINGS_SYMBOLE
        highlight_group = 'active_triggers_%s' % (WARNINGS)

    elif triggers == CRITICAL:
        triggers = zabbix.get_current_high_triggers()
        symbole = CRITICAL_SYMBOLE
        highlight_group = 'active_triggers_%s' % (CRITICAL)

    triggers_count = len(triggers)

    if 'current_count' in segment_info:
        if segment_info['current_count'] != triggers_count:
            delta = triggers_count - segment_info['current_count']
            if delta == triggers_count:
                delta = 0
            sign = SYMBOLE_NEGATIVE if delta < 0 else SYMBOLE_ADD

            segment_info['last_delta'] = delta
            segment_info['current_count'] = triggers_count
        else:
            delta = segment_info['last_delta']

    else:
        segment_info['current_count'] = triggers_count
        sign = SYMBOLE_NODIFF
        delta = 0

    return [{
        'contents': u"%s %s%s%s" % (
            symbole, triggers_count, sign, delta),
        'highlight_group': [highlight_group]
    }]
