__author__ = 'Matthew Campbell'


class Progress():
  """ Convert a set unit, value, and range to progress representation. """
  __current_value = None
  __max_value = None
  __units = None
  __completed_character = None
  __remaining_character = None
  __progress_bar_length = None

  def __init__(self):
    self.__current_value = 0
    self.__progress_bar_length = 10
    self.__completed_character = '|'
    self.__remaining_character = '.'
    self.__units = ''

  def __scale(self):
    """ Map the input value and range to the outputted progress bar. """
    in_min = 0
    in_max = self.__max_value
    out_min = 0
    out_max = self.__progress_bar_length

    # Figure out how 'wide' each range is
    in_span = in_max - in_min
    out_span = out_max - out_min

    # Convert the input range into a 0-1 range (float)
    value_scaled = float(self.__current_value - in_min) / float(in_span)

    # Convert the 0-1 range into a value in the output range.
    return (int(out_min + (value_scaled * out_span)), int(out_span))

  def __str__(self):
    """ Return progress information including a progress bar """
    if self.__current_value is None or self.__max_value is None:
      raise IOError("\n\nError: The current value and max value of a progress object must be set before printing.\n")

    completed, max = self.__scale()
    remaining = max - completed

    c = self.__completed_character
    r = self.__remaining_character
    u = self.__units

    cc = self.__current_value
    m = self.__max_value

    return "{:03d}/{:03d}, {}/{} {} [{}{}]".format(completed, max, cc, m, u, completed * c, remaining * r)

  def set_value(self, value):
    """ The current progress value. """
    if not isinstance(value, int):
      raise IOError("\n\nError: progress.set_value(int) parameter must be of type int.\n")
    assert isinstance(value, int)
    self.__current_value = value

  def set_max(self, value):
    """ Set the max for the progress value. The min value is always zero. """
    if not isinstance(value, int):
      raise IOError("\n\nError: progress.set_max(int) parameter must be of type int.\n")
    assert isinstance(value, int)
    self.__max_value = value

  def set_units(self, units):
    """ Set the units to be displayed next to the current value. """
    self.__units = str(units)

  def set_length(self, length):
    """ Set the character length of the progress bar. """
    if not isinstance(length, int):
      raise IOError("\n\nError: progress.set_length(int) parameter must be of type int.\n")
    if length > 100:
      length = 100

    self.__progress_bar_length = length

  def get_value(self):
    """ Get the current progress value """
    return self.__current_value

  def get_max(self):
    """ Get the max progress value. """
    return self.__max_value

