import sys, itertools, operator, unittest
from zflow import *

@component("IN", "IN", "OUT")
def printer(source, dest, out, spacer='\n'):
    fps = list(dest)
    for line in source:
        for fp in fps:
            fp.write(str(line))
            if spacer:
                fp.write(spacer)
    yield None

@component("IN", "OUT")
def reader(source, out):
    """ Almost might as well be a no-op. """
    for line in source:
        yield line

@component("IN", "OUT")
def numerate(source, out):
    for i, line in enumerate(source):
        yield i

@component('IN', 'OUT', 'OUT')
def split(source, out1, out2):
    for line in source:
        yield (out1, line)
        yield (out2, line)

@component("IN", "OUT")
def increment(source, out):
    for line in source:
        yield line+1

@component("IN", "OUT")
def inc(value, out):
    for line in value:
        yield line+1

@component("IN", "OUT")
def codify(source, out):
    for line in source:
        if isinstance(line, basestring):
            yield eval(line)
        else:
            yield line

from cStringIO import StringIO

@component("INA", "OUT")
def tabulate(source, out, pad=5):
    source = list(source)
    source = [list(s) for s in source]
    col_widths = [0 for x in range(len(source))]
    for i, col in enumerate(source):
        for entry in col:
            c_len = len(str(entry))
            if c_len > col_widths[i]:
                col_widths[i] = c_len
    col_widths = [c+pad for c in col_widths]
    for row in zip(*source):
        buf = StringIO()
        for width, item in zip(col_widths, row):
            buf.write(str(item).ljust(width))
        yield buf.getvalue()
 
@component("IN", "IN", "OUT")
def div(op1, op2, result):
    for a, b in itertools.izip(op1, op2):
        yield a / b

@component("IN", "IN", "OUT", "OUT")
def broken_crap(inp, carry, out, push):
    # Generator already executing? Hmm. Loops are very semantically foggy
    # I have no idea what this thing is supposed to do, but it doesn't work anyway.
    for num in inp:
        if not num:
            break
        yield (out, num+carry.next())
        yield (push, num-1)


@component("INA", "OUT")
def sum(ops, out):
    for nums in itertools.izip(*ops):
        yield reduce(operator.add, list(nums))

@component("IN", "OUTA")
def splitm(source, out, size=3):
    yield (out.size(), size)
    for s in source:
        for i in range(size):
            yield (out(i), s)

class ZFlowUnitTest(unittest.TestCase):
    def test_runner1(self):
        sum_r = runner(sum)
        inc1_r = runner(increment)
        inc1_r.connect_in('source', range(10))
        inc2_r = runner(increment)
        inc2_r.connect_in('source', range(10))
        sum_r.connect_in('ops', inc1_r('out'))
        sum_r.connect_in('ops', inc2_r('out'))

        buf = StringIO()
        print_r = runner(printer)
        print_r.connect_in('dest', iter([buf]))
        print_r.connect_in('source', sum_r('out'))
        list(print_r('out'))
        self.assertEquals(buf.getvalue(), '2\n4\n6\n8\n10\n12\n14\n16\n18\n20\n')

    def test_runner2(self):
        splitm_r = runner(splitm)
        splitm_r.connect_in('source', [1, 2, 3, 4, 5])
        p = [runner(printer) for i in range(3)]
        buf = StringIO()
        for i, pp in enumerate(p):
            pp.connect_in('source', splitm_r('out', i))
            pp.connect_in('dest', iter([buf]))
        x = itertools.izip(*[pp('out') for pp in p])
        list(x)
        self.assertEquals(buf.getvalue(), '1\n2\n3\n4\n5\n1\n2\n3\n4\n5\n1\n2\n3\n4\n5\n')

    def test_runner3(self):
        ## CONNECTING AN OUT ARRAY TO AN IN ARRAY
        splitm_r = runner(splitm)
        splitm_r.connect_in('source', range(10))
        sum_r = runner(sum)
        sum_r.connect_in('ops', splitm_r('out'))
        x = list(sum_r('out'))
        self.assertEquals(x, [0, 3, 6, 9, 12, 15, 18, 21, 24, 27])

    def test_runner4(self):
        splitm_r = runner(splitm, {'size' : 3})
        sum_r = runner(sum)
        inc_r = runner(inc)
        sum_r.connect_in('ops', splitm_r('out', 0))
        sum_r.connect_in('ops', splitm_r('out', 1))
        sum_r.connect_in('ops', inc_r('out'))
        inc_r.connect_in('value', splitm_r('out', 2))
        splitm_r.connect_in('source', range(5))

        x = list(sum_r('out'))
        self.assertEquals(x, [1, 4, 7, 10, 13])

    def test_network0(self):
        triple1 = network(
            name='triple1',
            mesh = (
                ('splitm', splitm, ("out", 0), "sum", sum, ("ops", 0)),
                ('splitm', splitm, ("out", 1), "sum", sum, ("ops", 1)),
                ('splitm', splitm, ("out", 2), "inc", inc, "value"),
                ('inc',    inc,    'out',      'sum', sum, ('ops', 2))),
            port_order = ('source', 'out'),
            control_values = {
                ('splitm', 'size') : 3},
            in_port_defs = {
                'source' : ("splitm", "source")},
            out_port_defs = {
                'out' : ("sum", "out")})

        t1 = runner(triple1)
        t1.connect_in('source', range(10))
        x = list(t1('out'))
        self.assertEquals(x, [1, 4, 7, 10, 13, 16, 19, 22, 25, 28])
 
    def test_network1(self):
        triple1 = network("triple1",
            [
                ["splitm", splitm, ("out", 0), "sum", sum, ("ops", 0)],
                ["splitm", splitm, ("out", 1), "sum", sum, ("ops", 1)],
                ["splitm", splitm, ("out", 2), "inc", inc, "value"],
                ["inc",    inc,    "out",      "sum", sum, ("ops", 2)]
            ],
            ["source", "out"],
            {"source" : ("splitm", "source")},
            {"out" : ("sum", "out")},
            {"size" : ("splitm", "size")}
        )

        t1 = runner(triple1, {'size' : 3})
        t1.connect_in('source', range(10))
        x = list(t1('out'))
        self.assertEquals(x, [1, 4, 7, 10, 13, 16, 19, 22, 25, 28])

        tharness = network("tharness",
            [
                ["split", split, "out1", "inc", inc, "value"],
                ["split", split, "out2", "triple1", triple1, "source"],
                ["inc", inc, "out", "tabulate", tabulate, ("source", 0)],
                ["triple1", triple1, "out", "tabulate", tabulate, ("source", 1)]
            ],
            ["source", "out"],
            {"source" : ("split", "source")},
            {"out" : ("tabulate", "out")},
            {"size" : ("triple1", "size")}
        )

        m = runner(tharness, dict(size=3))
        m.connect_in("source", range(10))
        x = list(m('out'))
        self.assertEquals(x[0], '1      1      ')
        self.assertEquals(x[1], '2      4      ')
        self.assertEquals(x[2], '3      7      ')
        self.assertEquals(x[3], '4      10     ')
        self.assertEquals(x[4], '5      13     ')
        self.assertEquals(x[5], '6      16     ')
        self.assertEquals(x[6], '7      19     ')
        self.assertEquals(x[7], '8      22     ')
        self.assertEquals(x[8], '9      25     ')
        self.assertEquals(x[9], '10     28     ')

    def test_network2(self):
        multiples = network("multiples",
            [
                ["splitm", splitm, "out", "sum", sum, "ops"]
            ],
            ["source", "out"],
            {"source" : ("splitm", "source")},
            {"out" : ("sum", "out")},
            {"size" : ("splitm", "size")}
        )

        m = runner(multiples, dict(size=10))
        m.connect_in("source", range(10))
        self.assertEquals(list(m("out")), [0, 10, 20, 30, 40, 50, 60, 70, 80, 90])

    def test_network3(self):
        sample_network = [
            ["codify",      codify,     "out", "increment", increment,  "source"],
            ["increment",   increment,  "out", "split",     split,      "source"],
            ["split",       split,      "out2", "numerate",  numerate,   "source"],
            ["numerate",    numerate,   "out", "tabulate",  tabulate,   "source"],
            ["split",       split,      "out1", "tabulate",  tabulate,   "source"]
        ]

        sample_ports = ['source', 'out']
        sample_in_ports = {'source' : ("codify", "source")}
        sample_out_ports = {'out' : ("tabulate", "out")}

        samplex = network('samplex',
            sample_network,
            sample_ports,
            sample_in_ports,
            sample_out_ports)

        sample_r = runner(samplex)
        sample_r.connect_in('source', [1, 2, 3, 4, 5])

        g = list(sample_r('out'))
        self.assertEquals(g, ['0     2     ', '1     3     ', '2     4     ', '3     5     ', '4     6     '])

    def test_network4(self):
        sample = network('sample', 
            [["reader", reader, "out", "printer", printer, "source"]],
            ['s', 'dest', 'out'],
            {'s' : ("reader", "source"), "dest" : ("printer", 'dest')},
            {'out' : ('printer', 'out')},
            {'spacer' : ('printer', 'spacer')}
        )

        buf = StringIO()
        sample_r = runner(sample, dict(spacer='...\n'))
        sample_r.connect_in('s', [1, 2, 3, 4, 5])
        sample_r.connect_in('dest', iter([buf]))

        g = sample_r('out')
        g.next()
        self.assertEquals(buf.getvalue(), '1...\n2...\n3...\n4...\n5...\n')

if __name__ == '__main__':
    unittest.main()
