import inspect, sys, itertools
import itertools
import sys

from pprint import pprint

__all__ = ['component', 'runner', 'pprint_component']

def component(*port_specs, **xtra_args):
    """ The component decorator converts a function into a dataflow
    component with specified IN ports and OUT ports given in the port_specs.
    Keyword arguments are control ports which are used to provide traditional
    python-style function arguments to the procedure.
    """
    def decor(fun):
        arg_specs = inspect.getargspec(fun)
        match = zip(port_specs, arg_specs[0])
        name = xtra_args.get('name', fun.__name__)
        in_ports = [m[1] for m in match if m[0] == 'IN']
        out_ports = [m[1] for m in match if m[0] == 'OUT']
        in_array_ports = [m[1] for m in match if m[0] == 'INA']
        out_array_ports = [m[1] for m in match if m[0] == 'OUTA']
        controls = arg_specs[0][len(port_specs):]

        if 'generator_port' in xtra_args:
            generator_port = xtra_args['generator_port']
            if generator_port not in out_ports:
                raise ValueError, ("The given generator port %r does not match "
                    "a port with an OUT port specification" % generator_port)
        elif len(out_ports) == 1:
            generator_port = out_ports[0]
        else:
            generator_port = None

        fun.zf_component = True
        fun.zf_name = name
        fun.zf_generator_port = generator_port
        fun.zf_in_ports = in_ports
        fun.zf_out_ports = out_ports
        fun.zf_in_array_ports = in_array_ports
        fun.zf_out_array_ports = out_array_ports
        fun.zf_port_order = arg_specs[0][:len(port_specs)]
        fun.zf_controls = controls
        return fun
    return decor

def pprint_component(comp, stream=sys.stdout):
    """ Debug procedure to dump out all the configuration on a component.
    """
    print >>stream, comp.zf_name
    if comp.__doc__:
        print >>stream, comp.__doc__
        print >>stream, ''

    print >>stream, "PORTS"
    print >>stream, "-"*80
    pmap = [
        ("IN", comp.zf_in_ports),
        ("OUT", comp.zf_out_ports),
        ("IN ARRAY", comp.zf_in_array_ports),
        ("OUT ARRAY", comp.zf_out_array_ports)
    ]
    for port in comp.zf_port_order:
        for s, v in pmap:
            if port in v:
                print >>stream, s.ljust(10),
                break
        else:
            print >>stream, "UNKNOWN".ljust(10),
        print >>stream, port
    if comp.zf_generator_port:
        print >>stream, ''
        print >>stream, "GENERATOR PORT: %s" % comp.zf_generator_port
    if comp.zf_controls:
        print >>stream, '\nCONTROLS'
        print >>stream, "-"*80
        for c in comp.zf_controls:
            print c

## The port number classes are used to provide a communication protocol
## for components to yield back values that are send to out ports
p_counter = itertools.count(1)
class PortNumber(int):
   def __new__(self):
        global p_counter
        n = p_counter.next()
        if n >= 1048576:
            p_counter = itertools.count(1)
            n = p_counter.next()
        return int.__new__(self, n)

class ArrayPortNumber(int):
    def __new__(self):
        global p_counter
        n = p_counter.next()
        if n >= 1048576:
            p_counter = itertools.count(1)
            n = p_counter.next()
        return int.__new__(self, n)

    def size(self):
        return ArrayPortSizeSpec(self)

    def __call__(self, idx):
        return ArrayPortSpec(self, idx)

class ArrayPortSpec(object):
    def __init__(self, num, idx):
        self.port = num
        self.idx = idx

class ArrayPortSizeSpec(object):
    def __init__(self, port):
        self.port = port

class runner(object):
    """Provides a run harness for a component."""
    def __init__(self, comp, commands={}):
        self.comp = comp
        self.out_buffer = dict([(n, []) for n in comp.zf_out_ports])
        self.out_lookup = {}
        self.out_array_lookup = {}
        self.in_gens = {}
        self.in_gen_arrays = {}
        self.arg_list = []
        self.build_arg_list()
        self.generator = self.comp(*self.arg_list, **commands)

    def input_stream(self, name):
        # We have to do this lazy-style so we can have loops in the graph
        #XXX: After doing the code, the loops don't work anyway, but we're
        #     keeping this construct.
        def closure():
            if name in self.in_gens:
                for value in self.in_gens[name]:
                    yield value
            elif name in self.in_gen_arrays:
                for value in self.in_gen_arrays[name]:
                    yield value
            else:
                raise ValueError, ("Invalid IN port %r on component %r. "
                    "Valid Ports: %s" %
                    (name, self.comp.zf_name, self.in_gens.keys() +
                                              self.in_gen_arrays.keys()))
        return closure()

    def connect_in(self, name, value):
        if name in self.comp.zf_in_ports:
            self.in_gens[name] = value
        elif name in self.comp.zf_in_array_ports:
            # We are directly assigning an array to the port. the zs_is_array
            # flag tells us so
            if getattr(value, 'zf_is_array', False):
                self.in_gen_arrays[name] = value
                return
            # We are appending connections to the array port in sequence as 
            # they come in
            if name not in self.in_gen_arrays:
                self.in_gen_arrays[name] = []
            self.in_gen_arrays[name].append(value)
        else:
            raise ValueError, "No port %r on component %r" % (
                                name, self.comp.zf_name)

    def build_arg_list(self):
        args = []
        comp = self.comp
        for port in comp.zf_port_order:
            if port in comp.zf_in_ports or port in comp.zf_in_array_ports:
                args.append(self.input_stream(port))
            elif port in comp.zf_out_ports:
                num = PortNumber()
                self.out_lookup[num] = port
                args.append(num)
            elif port in comp.zf_out_array_ports:
                num = ArrayPortNumber()
                self.out_lookup[num] = port
                args.append(num)
            else:
                raise ValueError, ("Component %r include a port order for "
                    "port %r, but the port is not registered as an IN port "
                    "or OUT port" % (comp.zf_name, port))
        self.arg_list = args

    def buffer_empty(self, out_port, idx):
        """Are there any results for us to yield?"""
        is_out_array_port = out_port in self.comp.zf_out_array_ports
        if is_out_array_port:
            # We are scanning a specific out array port index
            if idx != -1:
                if out_port not in self.out_array_lookup:
                    return True
                elif not self.out_array_lookup[out_port][idx]:
                    return True
                else:
                    return False
            # We are scanning an entire out array port. As soon as the
            # size has been set, we can return our value
            else:
                if out_port not in self.out_array_lookup:
                    return True
                else:
                    return False
        else:
            return not bool(self.out_buffer[out_port])

    def yield_value(self, out_port, idx):
        """ Provide the value for the given outport and out array index. """
        is_out_array_port = out_port in self.comp.zf_out_array_ports
        if is_out_array_port:
            if idx != -1:
                return self.out_array_lookup[out_port][idx].pop(0), False
            else:
                return [self(out_port, i) for i in 
                        range(len(self.out_array_lookup[out_port]))], True
        else:
            return self.out_buffer[out_port].pop(0), False

    def __call__(self, out_port, idx=-1):
        """ This is the runtime engine """
        ii = isinstance
        def run():
            while True:
                while self.buffer_empty(out_port, idx):
                    result = self.generator.next()
                    if (ii(result, tuple) and len(result) == 2
                        and ii(result[0], PortNumber)):
                        this_port = self.out_lookup[result[0]]
                        line = result[1]
                        self.out_buffer[this_port].append(line)
                    elif (ii(result, tuple) and len(result) == 2
                          and ii(result[0], ArrayPortSizeSpec)):
                        this_port = self.out_lookup[result[0].port]
                        self.out_array_lookup[out_port] = [
                            [] for i in range(result[1])]
                    elif (ii(result, tuple) and len(result) == 2 
                          and ii(result[0], ArrayPortSpec)):
                        # Pushing a value in an array output index
                        this_port = self.out_lookup[result[0].port]
                        index = result[0].idx
                        if this_port not in self.out_array_lookup:
                            raise ValueError, ("Encountered OUT array "
                                "port value before a size has been specified "
                                "on component %r, port %r" % 
                                (self.comp.zf_name, this_port))
                        self.out_array_lookup[this_port][index].append(
                            result[1])
                    elif self.comp.zf_generator_port:
                        self.out_buffer[self.comp.zf_generator_port].append(
                            result)
                    else:
                        raise ValueError, ("component %r yielded invalid "
                            "value %r. Please provide a generator_port or "
                            "yield (port, value)" % (self.comp.zf_name, result))
                value, stop = self.yield_value(out_port, idx)
                if stop:
                    for v in value:
                        yield v
                    return
                else:
                    yield value

        # We have to put this here so we can detect in on in_connect to 
        # hook up out arrays to in arrays properly
        gen = run()
        if out_port in self.comp.zf_out_array_ports and idx == -1:
            return OutArrayIterator(gen)
        else:
            return gen

class OutArrayIterator(object):
    """ We have to make this wrapper because we can't assign zf_is_array 
    to a generator object because its attributes are read only. """
    def __init__(self, gen):
        self.gen = gen
        self.zf_is_array = True
    def __iter__(self):
        return self.gen
