import sys
import itertools
import operator
import unittest

from cStringIO import StringIO

from zflow import *

@component("IN", "IN", "OUT")
def printer(source, dest, out, spacer='\n'):
    fps = list(dest)
    for line in source:
        for fp in fps:
            fp.write(str(line))
            if spacer:
                fp.write(spacer)
    yield None

@component("IN", "OUT")
def reader(source, out):
    """ Almost might as well be a no-op. """
    for line in source:
        yield line

@component("IN", "OUT")
def numerate(source, out):
    for i, line in enumerate(source):
        yield i

@component('IN', 'OUT', 'OUT')
def split(source, out1, out2):
    for line in source:
        yield (out1, line)
        yield (out2, line)

@component("IN", "OUT")
def increment(source, out):
    for line in source:
        yield line+1

@component("IN", "OUT")
def inc(value, out):
    for line in value:
        yield line+1

@component("IN", "OUT")
def codify(source, out):
    for line in source:
        if isinstance(line, basestring):
            yield eval(line)
        else:
            yield line

@component("INA", "OUT")
def tabulate(source, out, pad=5):
    source = list(source)
    source = [list(s) for s in source]
    col_widths = [0 for x in range(len(source))]
    for i, col in enumerate(source):
        for entry in col:
            c_len = len(str(entry))
            if c_len > col_widths[i]:
                col_widths[i] = c_len
    col_widths = [c+pad for c in col_widths]
    for row in zip(*source):
        buf = StringIO()
        for width, item in zip(col_widths, row):
            buf.write(str(item).ljust(width))
        yield buf.getvalue()
 
@component("IN", "IN", "OUT")
def div(op1, op2, result):
    for a, b in itertools.izip(op1, op2):
        yield a / b

@component("IN", "IN", "OUT", "OUT")
def broken_crap(inp, carry, out, push):
    # Generator already executing? Hmm. Loops are very semantically foggy
    # I have no idea what this thing is supposed to do, but it doesn't work
    # anyway.
    for num in inp:
        if not num:
            break
        yield (out, num+carry.next())
        yield (push, num-1)

@component("INA", "OUT")
def sum(ops, out):
    for nums in itertools.izip(*ops):
        yield reduce(operator.add, list(nums))

@component("IN", "OUTA")
def splitm(source, out, size=3):
    yield (out.size(), size)
    for s in source:
        for i in range(size):
            yield (out(i), s)

class ZFlowTestCase(unittest.TestCase):
    def test_runner1(self):
        sum_r = runner(sum)
        inc1_r = runner(increment)
        inc1_r.connect_in('source', range(10))
        inc2_r = runner(increment)
        inc2_r.connect_in('source', range(10))
        sum_r.connect_in('ops', inc1_r('out'))
        sum_r.connect_in('ops', inc2_r('out'))

        buf = StringIO()
        print_r = runner(printer)
        print_r.connect_in('dest', iter([buf]))
        print_r.connect_in('source', sum_r('out'))
        list(print_r('out'))
        self.assertEquals(buf.getvalue(), 
            '2\n4\n6\n8\n10\n12\n14\n16\n18\n20\n')

    def test_runner2(self):
        splitm_r = runner(splitm)
        splitm_r.connect_in('source', [1, 2, 3, 4, 5])
        p = [runner(printer) for i in range(3)]
        buf = StringIO()
        for i, pp in enumerate(p):
            pp.connect_in('source', splitm_r('out', i))
            pp.connect_in('dest', iter([buf]))
        x = itertools.izip(*[pp('out') for pp in p])
        list(x)
        self.assertEquals(buf.getvalue(), 
            '1\n2\n3\n4\n5\n1\n2\n3\n4\n5\n1\n2\n3\n4\n5\n')

    def test_runner3(self):
        ## CONNECTING AN OUT ARRAY TO AN IN ARRAY
        splitm_r = runner(splitm)
        splitm_r.connect_in('source', range(10))
        sum_r = runner(sum)
        sum_r.connect_in('ops', splitm_r('out'))
        x = list(sum_r('out'))
        self.assertEquals(x, [0, 3, 6, 9, 12, 15, 18, 21, 24, 27])

    def test_runner4(self):
        splitm_r = runner(splitm, {'size' : 3})
        sum_r = runner(sum)
        inc_r = runner(inc)
        sum_r.connect_in('ops', splitm_r('out', 0))
        sum_r.connect_in('ops', splitm_r('out', 1))
        sum_r.connect_in('ops', inc_r('out'))
        inc_r.connect_in('value', splitm_r('out', 2))
        splitm_r.connect_in('source', range(5))

        x = list(sum_r('out'))
        self.assertEquals(x, [1, 4, 7, 10, 13])

if __name__ == '__main__':
    unittest.main()
