import difflib
import itertools
import re
from collections import defaultdict
from django.conf import settings
from django.utils.datastructures import SortedDict
from zumanji.github import github
from zumanji.models import Revision, Test, TestData

HISTORICAL_POINTS = 25

REVISION_RE = re.compile(r'^[A-Za-z0-9]{40}$')


def is_revision(value):
    return REVISION_RE.match(value)


def is_github_revision(project, value):
    return is_revision(value) and '/' in project.label


def get_trace_data(test, previous_test=None):
    if previous_test:
        try:
            previous_trace = previous_test.testdata_set.get(key='trace').data
        except TestData.DoesNotExist:
            previous_trace = []
    else:
        previous_trace = []

    try:
        trace = test.testdata_set.get(key='trace').data
    except TestData.DoesNotExist:
        trace = []

    if not (trace or previous_trace):
        return {}

    previous_trace = SortedDict(('%s_%s' % (x, c['id']), c) for x, c in enumerate(previous_trace))
    trace = SortedDict(('%s_%s' % (x, c['id']), c) for x, c in enumerate(trace))

    seqmatch = difflib.SequenceMatcher()
    seqmatch.set_seqs(previous_trace.keys(), trace.keys())

    trace_diff = (
        {'test': previous_test, 'calls': []},  # left
        {'test': test, 'calls': []},  # left
    )
    for tag, i1, i2, j1, j2 in seqmatch.get_opcodes():
        if tag in ('equal', 'replace'):
            for key in previous_trace.keys()[i1:i2]:
                trace_diff[0]['calls'].append((tag, key, previous_trace[key]))
            for key in trace.keys()[j1:j2]:
                trace_diff[1]['calls'].append((tag, key, trace[key]))
        elif tag == 'delete':
            for key in previous_trace.keys()[i1:i2]:
                trace_diff[0]['calls'].append((tag, key, previous_trace[key]))
                trace_diff[1]['calls'].append((tag, key, None))
        elif tag == 'insert':
            for key in trace.keys()[j1:j2]:
                trace_diff[0]['calls'].append((tag, key, None))
                trace_diff[1]['calls'].append((tag, key, trace[key]))
        else:
            raise ValueError(tag)

    all_calls = dict(previous_trace)
    all_calls.update(trace)

    return {
        'diff': trace_diff,
        'calls': all_calls,
        'num_diffs': sum(sum(1 for t, _, c in n['calls'] if t != 'equal') for n in trace_diff),
    }


def get_historical_data(build, test_list):
    previous_builds = []
    cur_build = build
    for point in xrange(HISTORICAL_POINTS):
        prev_build = cur_build.get_previous_build()
        if prev_build is None:
            break
        previous_builds.insert(0, prev_build.id)
        cur_build = prev_build

    previous_tests = list(Test.objects.filter(
        build__in=previous_builds,
        label__in=[t.label for t in test_list]
    )
    .order_by('-build__datetime'))

    historical = defaultdict(lambda: defaultdict(list))
    for test in itertools.chain(previous_tests, test_list):
        history_data = []
        for interface, _ in settings.ZUMANJI_CONFIG['call_types']:
            if interface not in test.data:
                history_data.append(0)
            else:
                interface_calls = test.data[interface].get('mean_calls')
                history_data.append(interface_calls)
        historical[test.label][test.build_id] = history_data

    padding = [(None, [])] * HISTORICAL_POINTS
    results = {}
    for test in test_list:
        results[test.id] = (padding + [
            (b, historical[test.label][b])
            for b in (previous_builds + [test.build_id])
        ])[-HISTORICAL_POINTS:]

    return results


def get_changes(previous_build, objects):
    if not (previous_build and objects):
        return {}

    qs = previous_build.test_set.filter(
        label__in=[o.label for o in objects],
    ).select_related('parent')

    previous_build_objects = dict((o.label, o) for o in qs)
    changes = dict()

    # {group: [{notes: notes, type: type}]}
    for obj in objects:
        last_obj = previous_build_objects.get(obj.label)
        obj_changes = {
            'interfaces': {},
            'status': 'new' if last_obj is None else None,
        }
        if last_obj:
            data = obj.data
            last_obj_data = last_obj.data
            for interface, _ in settings.ZUMANJI_CONFIG['call_types']:
                if interface in data:
                    current = data[interface].get('mean_calls', 0)
                else:
                    current = 0

                if interface in last_obj_data:
                    previous = last_obj_data[interface].get('mean_calls', 0)
                else:
                    previous = 0

                change = current - previous
                if change == 0:
                    continue

                obj_changes['interfaces'][interface] = {
                    'current': current,
                    'previous': previous,
                    'change': '+%s' % change if change > 0 else str(change),
                    'type': 'increase' if change > 0 else 'decrease',
                }

        if obj_changes['status'] != 'new' and not obj_changes['interfaces']:
            continue

        changes[obj] = obj_changes

    return sorted(changes.iteritems(),
        key=lambda x: sum(int(i['change']) for i in x[1]['interfaces'].values()),
        reverse=True)


def get_git_changes(build, previous_build):
    project = build.project

    results = github.compare_commits(project.github_user, project.github_repo,
        previous_build.revision.label, build.revision.label)
    commits = []
    for commit in results['commits']:
        revision = Revision.get_or_create(build.project, commit['sha'])
        commits.append(revision)
    return commits
